<?php
/**
 * ============================================================
 * أوامر السوبر موزع (Super Distributor Commands)
 * ============================================================
 * 
 * يحتوي جميع أوامر السوبر موزع:
 * - عرض الحالة والحسابات
 * - توليد أكواد التفعيل
 * - البث لزبائنه
 * - عرض الحدود المتبقية
 * 
 * ============================================================
 * ملاحظات:
 * ============================================================
 * - الأوامر الحساسة تتطلب اشتراك بالقناة
 * - يتم فحص حدود توليد الأكواد قبل التوليد
 * - البث يحترم صلاحيات can_broadcast و broadcast_allowed_types
 */

class SuperDistributorCommands
{
    /**
     * معالجة الأمر الوارد
     */
    public static function handle(string $text, int $fromId, int $chatId, array $message, array $superdist): void
    {
        // تحليل الأمر
        $parts = preg_split('/\s+/', $text, 3);
        $command = strtolower($parts[0]);
        $arg1 = $parts[1] ?? '';
        $arg2 = $parts[2] ?? '';
        
        switch ($command) {
            // عرض الحسابات
            case '/my_accounts':
            case '/myaccounts':
                self::cmdMyAccounts($chatId, $superdist);
                break;
            
            // حالة السوبر موزع
            case '/superdist_status':
            case '/mystatus':
                self::cmdMyStatus($chatId, $superdist);
                break;
            
            // الحدود المتبقية
            case '/my_remaining':
            case '/remaining':
                self::cmdMyRemaining($chatId, $superdist);
                break;
            
            // توليد كود واحد
            case 'gen_code':
            case '/gen_code':
                self::cmdGenCode($chatId, $fromId, $superdist, $arg1);
                break;
            
            // توليد مجموعة أكواد
            case 'gen_codes':
            case '/gen_codes':
                self::cmdGenCodes($chatId, $fromId, $superdist, $arg1, $arg2);
                break;
            
            // عرض أكوادي
            case '/my_codes':
            case '/mycodes':
                self::cmdMyCodes($chatId, $superdist, $arg1, (int)$arg2 ?: 1);
                break;
            
            // فحص كود
            case '/check_code':
                self::cmdCheckCode($chatId, $superdist, $arg1);
                break;
            
            // البث لزبائني
            case '/broadcast_my_users':
            case '/broadcast':
                self::cmdBroadcast($chatId, $fromId, $superdist);
                break;
            
            // البث لزبائن حساب معين
            case '/broadcast_my_account':
                self::cmdBroadcastAccount($chatId, $fromId, $superdist, $arg1);
                break;
            
            default:
                Telegram::sendMessage($chatId, "❓ أمر غير معروف.\n\nاستخدم /help لعرض الأوامر.");
        }
    }
    
    // ================================================================
    // أوامر عامة
    // ================================================================
    
    /**
     * رسالة الترحيب
     */
    public static function cmdStart(int $chatId, array $superdist): void
    {
        $text = Replies::getForSuperDist('WELCOME_SUPERDIST', $superdist['id'], [
            'name' => $superdist['display_name'],
        ]);
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * عرض المساعدة
     */
    public static function cmdHelp(int $chatId): void
    {
        $text = "📋 <b>أوامر الموزع</b>\n\n";
        
        // عرض المعلومات
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "<b>📊 عرض المعلومات</b>\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        
        $text .= "<b>1️⃣ حساباتك المرتبطة</b>\n";
        $text .= "<code>/my_accounts</code>\n";
        $text .= "📝 <b>الشرح:</b> عرض جميع الحسابات المرتبطة بك\n";
        $text .= "✅ <b>النتيجة:</b>\n<i>🟢 حساب VIP\n   └ Slug: vip-acc\n   └ متبقي: 25 يوم</i>\n\n";
        
        $text .= "<b>2️⃣ حالتك وإحصائياتك</b>\n";
        $text .= "<code>/mystatus</code>\n";
        $text .= "📝 <b>الشرح:</b> عرض حالتك وإحصائيات أكوادك\n";
        $text .= "✅ <b>النتيجة:</b>\n<i>• الحالة: 🟢 نشط\n• أكواد اليوم: 5\n• إجمالي الزبائن: 120</i>\n\n";
        
        $text .= "<b>3️⃣ الحدود المتبقية</b>\n";
        $text .= "<code>/my_remaining</code>\n";
        $text .= "📝 <b>الشرح:</b> عرض حدود توليد الأكواد المتبقية\n";
        $text .= "✅ <b>النتيجة:</b>\n<i>📊 الحدود المتبقية:\n• يومي: 8/10\n• أسبوعي: 45/50</i>\n\n";
        
        // توليد الأكواد
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "<b>🔑 توليد الأكواد</b>\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        
        $text .= "<b>4️⃣ توليد كود واحد</b>\n";
        $text .= "<code>gen_code [slug]</code>\n";
        $text .= "📝 <b>الشرح:</b> توليد كود تفعيل واحد\n";
        $text .= "📌 <b>مثال:</b>\n<code>gen_code vip-acc</code>\n";
        $text .= "✅ <b>النتيجة:</b>\n<i>🎫 كود التفعيل:\nACT-VIP-A1B2C3D4</i>\n";
        $text .= "⚠️ <b>ملاحظات:</b> يجب أن يكون الحساب مرتبط بك ونشط\n\n";
        
        $text .= "<b>5️⃣ توليد مجموعة أكواد</b>\n";
        $text .= "<code>gen_codes [slug] [count]</code>\n";
        $text .= "📝 <b>الشرح:</b> توليد عدة أكواد دفعة واحدة\n";
        $text .= "📌 <b>مثال:</b>\n<code>gen_codes vip-acc 5</code>\n";
        $text .= "✅ <b>النتيجة:</b>\n<i>✅ تم توليد 5 أكواد:\n1. ACT-VIP-...\n2. ACT-VIP-...</i>\n";
        $text .= "⚠️ <b>ملاحظات:</b> الحد الأقصى يعتمد على حدودك اليومية\n\n";
        
        $text .= "<b>6️⃣ عرض أكوادك</b>\n";
        $text .= "<code>/my_codes [slug] [page]</code>\n";
        $text .= "📝 <b>الشرح:</b> عرض قائمة الأكواد التي أنشأتها\n";
        $text .= "📌 <b>مثال:</b>\n<code>/my_codes vip-acc 1</code>\n\n";
        
        $text .= "<b>7️⃣ فحص كود</b>\n";
        $text .= "<code>/check_code [code]</code>\n";
        $text .= "📝 <b>الشرح:</b> فحص حالة كود معين\n";
        $text .= "📌 <b>مثال:</b>\n<code>/check_code ACT-VIP-A1B2C3D4</code>\n";
        $text .= "✅ <b>النتيجة:</b>\n<i>🎫 معلومات الكود:\n• الحالة: ✅ غير مستخدم\n• الحساب: VIP</i>\n\n";
        
        // البث
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "<b>📢 البث</b>\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        
        $text .= "<b>8️⃣ بث لجميع زبائنك</b>\n";
        $text .= "<code>/broadcast</code>\n";
        $text .= "📝 <b>الشرح:</b> إرسال رسالة لجميع زبائنك\n";
        $text .= "✅ <b>النتيجة:</b>\n<i>📢 أرسل المحتوى...\n👥 عدد الزبائن: 50\n📝 المحتوى المسموح: text, photos</i>\n";
        $text .= "⚠️ <b>ملاحظات:</b> يجب أن تملك صلاحية البث\n\n";
        
        $text .= "<b>9️⃣ بث لزبائن حساب معين</b>\n";
        $text .= "<code>/broadcast_my_account [slug]</code>\n";
        $text .= "📝 <b>الشرح:</b> إرسال رسالة لزبائن حساب محدد\n";
        $text .= "📌 <b>مثال:</b>\n<code>/broadcast_my_account vip-acc</code>\n\n";
        
        // كود الاشتراك
        $text .= "━━━━━━━━━━━━━━━━━━━━\n";
        $text .= "<b>🎟️ كود الاشتراك</b>\n";
        $text .= "━━━━━━━━━━━━━━━━━━━━\n\n";
        
        $text .= "<b>🔟 تفعيل/تجديد حساباتك</b>\n";
        $text .= "📝 <b>الشرح:</b> أرسل كود SD-... لتفعيل أو تجديد حساباتك\n";
        $text .= "📌 <b>مثال:</b>\n<code>SD-ABCD-1234-EFGH</code>\n";
        $text .= "✅ <b>النتيجة:</b>\n<i>✅ تم التفعيل!\n• الحسابات: VIP, Premium\n• المدة: 30 يوم</i>\n";
        
        Telegram::sendLongMessage($chatId, $text);
    }
    
    // ================================================================
    // الحسابات والحالة
    // ================================================================
    
    /**
     * عرض حساباتي المرتبطة
     */
    public static function cmdMyAccounts(int $chatId, array $superdist): void
    {
        $accounts = Db::fetchAll(
            "SELECT sa.*, a.slug, a.name, a.status as account_status
             FROM superdist_accounts sa
             JOIN accounts a ON a.id = sa.account_id
             WHERE sa.superdist_id = ?
             ORDER BY sa.end_at DESC",
            [$superdist['id']]
        );
        
        if (empty($accounts)) {
            Telegram::sendMessage($chatId, Replies::get('ERROR_NO_ACTIVE_ACCOUNTS_SUPERDIST'));
            return;
        }
        
        $text = "📦 <b>حساباتك المرتبطة</b>\n\n";
        
        $hasActive = false;
        foreach ($accounts as $acc) {
            $isActive = strtotime($acc['end_at']) > time() && $acc['account_status'] === 'active';
            
            if ($isActive) {
                $hasActive = true;
                $icon = '🟢';
                $remaining = formatRemainingDays($acc['end_at']);
                $text .= "{$icon} <b>{$acc['name']}</b>\n";
                $text .= "   └ Slug: <code>{$acc['slug']}</code>\n";
                $text .= "   └ متبقي: {$remaining}\n";
                $text .= "   └ أكواد اليوم: {$acc['activations_today']}\n\n";
            } else {
                $icon = '🔴';
                $text .= "{$icon} <b>{$acc['name']}</b> (منتهي أو معطل)\n\n";
            }
        }
        
        if (!$hasActive) {
            $text .= "⚠️ لا تملك حسابات نشطة حالياً.\n";
        }
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * عرض حالتي
     */
    public static function cmdMyStatus(int $chatId, array $superdist): void
    {
        $statusIcons = [
            'active'    => '🟢 نشط',
            'suspended' => '🟡 موقوف',
            'banned'    => '🔴 محظور',
            'archived'  => '📦 مؤرشف',
        ];
        
        // إحصائيات
        $totalCodes = Db::count('activation_codes', 'superdist_id = ?', [$superdist['id']]);
        $usedCodes = Db::count('activation_codes', 'superdist_id = ? AND status = ?', [$superdist['id'], 'used']);
        $todayCodes = Db::count('activation_codes', 'superdist_id = ? AND DATE(created_at) = CURDATE()', [$superdist['id']]);
        $totalUsers = Db::count('user_accounts', 'superdist_id = ?', [$superdist['id']]);
        
        $broadcastStatus = $superdist['can_broadcast'] ? '✅ مسموح' : '❌ ممنوع';
        
        $text = "👤 <b>حالتك كموزع</b>\n\n";
        $text .= "• <b>الاسم:</b> {$superdist['display_name']}\n";
        $text .= "• <b>الحالة:</b> " . ($statusIcons[$superdist['status']] ?? $superdist['status']) . "\n";
        $allowedTypesStr = $superdist['broadcast_allowed_types'] ? implode(', ', json_decode($superdist['broadcast_allowed_types'], true) ?? []) : 'جميع الأنواع';
        $text .= "• <b>البث:</b> {$broadcastStatus} ({$allowedTypesStr})\n\n";
        $text .= "📊 <b>الإحصائيات:</b>\n";
        $text .= "• أكواد التفعيل: {$usedCodes}/{$totalCodes}\n";
        $text .= "• أكواد اليوم: {$todayCodes}\n";
        $text .= "• إجمالي الزبائن: {$totalUsers}\n";
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * عرض الحدود المتبقية
     */
    public static function cmdMyRemaining(int $chatId, array $superdist): void
    {
        // جلب الحسابات النشطة مع حدودها
        $accounts = Db::fetchAll(
            "SELECT sa.*, a.slug, a.name
             FROM superdist_accounts sa
             JOIN accounts a ON a.id = sa.account_id
             WHERE sa.superdist_id = ? AND sa.end_at > NOW() AND a.status = 'active'",
            [$superdist['id']]
        );
        
        if (empty($accounts)) {
            Telegram::sendMessage($chatId, "⚠️ لا تملك حسابات نشطة.");
            return;
        }
        
        $text = "📊 <b>الحدود المتبقية لتوليد الأكواد</b>\n\n";
        
        foreach ($accounts as $acc) {
            // NULL = بلا حد (غير محدود)، 0 = محظور، > 0 = الحد الفعلي
            $dailyLimit = $acc['activation_limit_daily'];
            $weeklyLimit = $acc['activation_limit_weekly'];
            $monthlyLimit = $acc['activation_limit_monthly'];
            
            $text .= "📦 <b>{$acc['name']}</b> (<code>{$acc['slug']}</code>)\n";
            
            $hasAnyLimit = false;
            
            if ($dailyLimit !== null) {
                $hasAnyLimit = true;
                if ($dailyLimit == 0) {
                    $text .= "   • يومي: ⛔ محظور\n";
                } else {
                    $dailyRemaining = max(0, $dailyLimit - ($acc['activations_today'] ?? 0));
                    $text .= "   • يومي: {$dailyRemaining}/{$dailyLimit}\n";
                }
            }
            
            if ($weeklyLimit !== null) {
                $hasAnyLimit = true;
                if ($weeklyLimit == 0) {
                    $text .= "   • أسبوعي: ⛔ محظور\n";
                } else {
                    $weeklyRemaining = max(0, $weeklyLimit - ($acc['activations_this_week'] ?? 0));
                    $text .= "   • أسبوعي: {$weeklyRemaining}/{$weeklyLimit}\n";
                }
            }
            
            if ($monthlyLimit !== null) {
                $hasAnyLimit = true;
                if ($monthlyLimit == 0) {
                    $text .= "   • شهري: ⛔ محظور\n";
                } else {
                    $monthlyRemaining = max(0, $monthlyLimit - ($acc['activations_this_month'] ?? 0));
                    $text .= "   • شهري: {$monthlyRemaining}/{$monthlyLimit}\n";
                }
            }
            
            if (!$hasAnyLimit) {
                $text .= "   • غير محدود ✨\n";
            }
            
            $text .= "\n";
        }
        
        Telegram::sendMessage($chatId, $text);
    }
    
    // ================================================================
    // توليد الأكواد
    // ================================================================
    
    /**
     * توليد كود واحد
     * الصيغة: gen_code <account_slug>
     */
    public static function cmdGenCode(int $chatId, int $fromId, array $superdist, string $accountSlug): void
    {
        if (empty($accountSlug)) {
            Telegram::sendMessage($chatId, "❌ الاستخدام:\n<code>gen_code [account_slug]</code>\n\nمثال:\n<code>gen_code vip-acc</code>\n\nاستخدم /my_accounts لعرض حساباتك.");
            return;
        }
        
        // فحص الاشتراك بالقناة
        if (!requireChannelSubscription($chatId, $fromId)) {
            return;
        }
        
        // جلب الحساب
        $account = getAccountBySlug(strtolower($accountSlug));
        if (!$account) {
            Telegram::sendMessage($chatId, "❌ الحساب غير موجود: {$accountSlug}");
            return;
        }
        
        // توليد الكود مسبقاً (لا يحتاج DB)
        $code = CodeGenerator::generateActivationCode();
        
        // حفظ الكود مع حماية من Race Condition
        try {
            Db::beginTransaction();
            
            // قفل السجل وفحص الحدود في نفس الـ transaction
            $sa = Db::fetchOne(
                "SELECT * FROM superdist_accounts 
                 WHERE superdist_id = ? AND account_id = ? 
                 FOR UPDATE",
                [$superdist['id'], $account['id']]
            );
            
            if (!$sa) {
                Db::rollback();
                Telegram::sendMessage($chatId, Replies::get('ERROR_ACCOUNT_NOT_ACTIVE'));
                return;
            }
            
            // التحقق من انتهاء الصلاحية
            if (strtotime($sa['end_at']) < time()) {
                Db::rollback();
                Telegram::sendMessage($chatId, "❌ انتهت صلاحية وصولك لهذا الحساب.");
                return;
            }
            
            // فحص الحدود بعد القفل
            $limitCheck = self::checkLimitsWithLockedRecord($sa, 1);
            if (!$limitCheck['allowed']) {
                Db::rollback();
                self::sendLimitError($chatId, $limitCheck, $superdist['id']);
                return;
            }
            
            Db::insert('activation_codes', [
                'code'         => $code,
                'account_id'   => $account['id'],
                'superdist_id' => $superdist['id'],
                'access_type'  => $account['default_access_type'],
                'access_days'  => $account['default_access_days'],
                'status'       => 'unused',
            ]);
            
            // تحديث العدادات
            Limits::incrementActivationCounters($superdist['id'], $account['id'], 1);
            
            Db::commit();
            
            // تسجيل الحدث
            Logger::codesGenerated($superdist['id'], $account['id'], 1, [$code]);
            
            Telegram::sendMessage($chatId, Replies::get('SUCCESS_CODE_GENERATED', [
                'code'         => $code,
                'account_name' => $account['name'],
            ]));
            
        } catch (Throwable $e) {
            Db::rollback();
            Logger::error('Failed to generate code', ['error' => $e->getMessage()]);
            Telegram::sendMessage($chatId, "❌ فشل توليد الكود. حاول مرة أخرى.");
        }
    }
    
    /**
     * توليد مجموعة أكواد
     * الصيغة: gen_codes <account_slug> <count>
     */
    public static function cmdGenCodes(int $chatId, int $fromId, array $superdist, string $accountSlug, string $countStr): void
    {
        $count = (int) $countStr;
        
        if (empty($accountSlug) || $count <= 0) {
            Telegram::sendMessage($chatId, "❌ الاستخدام:\n<code>gen_codes [account_slug] [count]</code>\n\nمثال:\n<code>gen_codes vip-acc 10</code>");
            return;
        }
        
        $maxCodes = config('MAX_CODES_PER_REQUEST', 50);
        if ($count > $maxCodes) {
            Telegram::sendMessage($chatId, "❌ الحد الأقصى لكل طلب: {$maxCodes} كود.");
            return;
        }
        
        // فحص الاشتراك بالقناة
        if (!requireChannelSubscription($chatId, $fromId)) {
            return;
        }
        
        // جلب الحساب
        $account = getAccountBySlug(strtolower($accountSlug));
        if (!$account) {
            Telegram::sendMessage($chatId, "❌ الحساب غير موجود: {$accountSlug}");
            return;
        }
        
        // توليد الأكواد مسبقاً (لا يحتاج DB)
        $codes = CodeGenerator::generateActivationCodes($count);
        
        try {
            Db::beginTransaction();
            
            // ============================================================
            // قفل السجل وفحص الحدود في نفس الـ transaction لمنع Race Condition
            // ============================================================
            $sa = Db::fetchOne(
                "SELECT * FROM superdist_accounts 
                 WHERE superdist_id = ? AND account_id = ? 
                 FOR UPDATE",
                [$superdist['id'], $account['id']]
            );
            
            if (!$sa) {
                Db::rollback();
                Telegram::sendMessage($chatId, Replies::get('ERROR_ACCOUNT_NOT_ACTIVE'));
                return;
            }
            
            // التحقق من انتهاء الصلاحية
            if (strtotime($sa['end_at']) < time()) {
                Db::rollback();
                Telegram::sendMessage($chatId, "❌ انتهت صلاحية وصولك لهذا الحساب.");
                return;
            }
            
            // إعادة فحص الحدود بعد القفل (مع البيانات المحدثة)
            $limitCheck = self::checkLimitsWithLockedRecord($sa, $count);
            if (!$limitCheck['allowed']) {
                Db::rollback();
                self::sendLimitError($chatId, $limitCheck, $superdist['id']);
                return;
            }
            
            foreach ($codes as $code) {
                Db::insert('activation_codes', [
                    'code'         => $code,
                    'account_id'   => $account['id'],
                    'superdist_id' => $superdist['id'],
                    'access_type'  => $account['default_access_type'],
                    'access_days'  => $account['default_access_days'],
                    'status'       => 'unused',
                ]);
            }
            
            // تحديث العدادات
            Limits::incrementActivationCounters($superdist['id'], $account['id'], $count);
            
            Db::commit();
            
            // تسجيل الحدث
            Logger::codesGenerated($superdist['id'], $account['id'], $count, $codes);
            
            // إرسال الأكواد كملف إذا كانت كثيرة
            if ($count > 5) {
                $codesText = implode("\n", $codes);
                $filename = "codes_{$account['slug']}_" . date('Ymd_His') . ".txt";
                $filepath = config('TEMP_PATH') . $filename;
                
                file_put_contents($filepath, $codesText);
                
                Telegram::sendDocument(
                    $chatId,
                    $filepath,
                    Replies::get('SUCCESS_CODES_GENERATED', [
                        'count'        => $count,
                        'account_name' => $account['name'],
                    ]),
                    $filename
                );
                
                unlink($filepath);
            } else {
                $codesFormatted = array_map(fn($c) => "<code>{$c}</code>", $codes);
                $text = "✅ تم توليد {$count} كود للحساب: {$account['name']}\n\n" . implode("\n", $codesFormatted);
                Telegram::sendMessage($chatId, $text);
            }
            
        } catch (Throwable $e) {
            Db::rollback();
            Logger::error('Failed to generate codes', ['error' => $e->getMessage()]);
            Telegram::sendMessage($chatId, "❌ فشل توليد الأكواد. حاول مرة أخرى.");
        }
    }
    
    /**
     * عرض أكوادي
     * الصيغة: /my_codes [account_slug] [page]
     */
    public static function cmdMyCodes(int $chatId, array $superdist, string $accountSlug, int $page): void
    {
        $perPage = 15;
        $offset = ($page - 1) * $perPage;
        
        $whereAccount = '';
        $params = [$superdist['id']];
        
        if (!empty($accountSlug)) {
            $account = getAccountBySlug(strtolower($accountSlug));
            if ($account) {
                $whereAccount = ' AND ac.account_id = ?';
                $params[] = $account['id'];
            }
        }
        
        $totalCodes = Db::fetchValue(
            "SELECT COUNT(*) FROM activation_codes ac WHERE ac.superdist_id = ?" . $whereAccount,
            $params
        );
        
        $totalPages = max(1, ceil($totalCodes / $perPage));
        
        if ($totalCodes == 0) {
            Telegram::sendMessage($chatId, "📋 لا توجد أكواد" . (!empty($accountSlug) ? " لهذا الحساب" : "") . ".");
            return;
        }
        
        $params[] = $perPage;
        $params[] = $offset;
        
        $codes = Db::fetchAll(
            "SELECT ac.*, a.slug, a.name as account_name
             FROM activation_codes ac
             JOIN accounts a ON a.id = ac.account_id
             WHERE ac.superdist_id = ?" . $whereAccount . "
             ORDER BY ac.created_at DESC
             LIMIT ? OFFSET ?",
            $params
        );
        
        $statusIcons = [
            'unused'   => '⚪',
            'used'     => '🟢',
            'expired'  => '🔴',
            'disabled' => '⛔',
        ];
        
        $text = "📋 <b>أكوادك</b> (ص{$page}/{$totalPages})\n\n";
        
        foreach ($codes as $code) {
            $icon = $statusIcons[$code['status']] ?? '❓';
            $text .= "{$icon} <code>{$code['code']}</code>\n";
            $text .= "   └ {$code['slug']} | " . date('m/d', strtotime($code['created_at'])) . "\n";
        }
        
        if ($totalPages > 1) {
            $text .= "\n─────────────────\n";
            $slugPart = !empty($accountSlug) ? " {$accountSlug}" : "";
            if ($page > 1) $text .= "⬅️ /my_codes{$slugPart} " . ($page - 1) . "\n";
            if ($page < $totalPages) $text .= "➡️ /my_codes{$slugPart} " . ($page + 1) . "\n";
        }
        
        Telegram::sendLongMessage($chatId, $text);
    }
    
    /**
     * فحص كود
     */
    public static function cmdCheckCode(int $chatId, array $superdist, string $code): void
    {
        if (empty($code)) {
            Telegram::sendMessage($chatId, "❌ الاستخدام: <code>/check_code [code]</code>");
            return;
        }
        
        $code = strtoupper(trim($code));
        
        $row = Db::fetchOne(
            "SELECT ac.*, a.slug, a.name as account_name, u.telegram_id as user_tg
             FROM activation_codes ac
             JOIN accounts a ON a.id = ac.account_id
             LEFT JOIN users u ON u.id = ac.used_by_user_id
             WHERE ac.code = ? AND ac.superdist_id = ?",
            [$code, $superdist['id']]
        );
        
        if (!$row) {
            Telegram::sendMessage($chatId, "❌ الكود غير موجود أو ليس لك.");
            return;
        }
        
        $statusText = [
            'unused'   => '⚪ غير مستخدم',
            'used'     => '🟢 مستخدم',
            'expired'  => '🔴 منتهي',
            'disabled' => '⛔ معطل',
        ];
        
        $text = "🔍 <b>معلومات الكود</b>\n\n";
        $text .= "• الكود: <code>{$row['code']}</code>\n";
        $text .= "• الحساب: {$row['account_name']} ({$row['slug']})\n";
        $text .= "• الحالة: " . ($statusText[$row['status']] ?? $row['status']) . "\n";
        $text .= "• أنشئ: " . formatDate($row['created_at']) . "\n";
        
        if ($row['status'] === 'used' && $row['user_tg']) {
            $text .= "• استخدمه: TG:{$row['user_tg']}\n";
            $text .= "• تاريخ الاستخدام: " . formatDate($row['used_at']) . "\n";
        }
        
        Telegram::sendMessage($chatId, $text);
    }
    
    // ================================================================
    // البث
    // ================================================================
    
    /**
     * بدء البث لزبائني
     */
    public static function cmdBroadcast(int $chatId, int $fromId, array $superdist): void
    {
        // التحقق من الصلاحية
        if (!$superdist['can_broadcast']) {
            Telegram::sendMessage($chatId, "❌ ليس لديك صلاحية البث.\n\nتواصل مع الإدارة لتفعيل هذه الميزة.");
            return;
        }
        
        // فحص الاشتراك بالقناة
        if (!requireChannelSubscription($chatId, $fromId)) {
            return;
        }
        
        // عدد الزبائن
        $usersCount = Db::count('user_accounts', 'superdist_id = ?', [$superdist['id']]);
        
        if ($usersCount == 0) {
            Telegram::sendMessage($chatId, "⚠️ ليس لديك زبائن حالياً.");
            return;
        }
        
        // بدء حالة البث
        State::startBroadcast($fromId, 'superdist', 'superdist_users', (string)$superdist['id']);
        
        // جلب الأنواع المسموحة
        $allowedTypesArr = json_decode($superdist['broadcast_allowed_types'] ?? '[]', true) ?: [];
        $allowedTypesStr = empty($allowedTypesArr) ? 'جميع الأنواع' : implode(', ', $allowedTypesArr);
        
        $text = Replies::get('INFO_BROADCAST_AWAITING_CONTENT');
        $text .= "\n\n👥 عدد الزبائن: {$usersCount}";
        $text .= "\n📝 المحتوى المسموح: {$allowedTypesStr}";
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * بدء البث لزبائن حساب معين
     * الصيغة: /broadcast_my_account <account_slug>
     */
    public static function cmdBroadcastAccount(int $chatId, int $fromId, array $superdist, string $accountSlug): void
    {
        // التحقق من الصلاحية
        if (!$superdist['can_broadcast']) {
            Telegram::sendMessage($chatId, "❌ ليس لديك صلاحية البث.");
            return;
        }
        
        if (empty($accountSlug)) {
            Telegram::sendMessage($chatId, "❌ الاستخدام:\n<code>/broadcast_my_account [account_slug]</code>\n\nمثال:\n<code>/broadcast_my_account vip-acc</code>");
            return;
        }
        
        // جلب الحساب
        $account = getAccountBySlug(strtolower($accountSlug));
        if (!$account) {
            Telegram::sendMessage($chatId, "❌ الحساب غير موجود: {$accountSlug}");
            return;
        }
        
        // التحقق من ربط الحساب
        $sa = self::getActiveSuperdistAccount($superdist['id'], $account['id']);
        if (!$sa) {
            Telegram::sendMessage($chatId, "❌ هذا الحساب غير مرتبط بك أو منتهي.");
            return;
        }
        
        // فحص الاشتراك بالقناة
        if (!requireChannelSubscription($chatId, $fromId)) {
            return;
        }
        
        // عدد زبائن هذا الحساب
        $usersCount = Db::count(
            'user_accounts',
            'superdist_id = ? AND account_id = ?',
            [$superdist['id'], $account['id']]
        );
        
        if ($usersCount == 0) {
            Telegram::sendMessage($chatId, "⚠️ ليس لديك زبائن على هذا الحساب.");
            return;
        }
        
        // بدء حالة البث مع فلتر الحساب
        State::startBroadcast($fromId, 'superdist', 'account_users', json_encode([
            'superdist_id' => $superdist['id'],
            'account_id'   => $account['id'],
        ]));
        
        // جلب الأنواع المسموحة
        $allowedTypesArr = json_decode($superdist['broadcast_allowed_types'] ?? '[]', true) ?: [];
        $allowedTypesStr = empty($allowedTypesArr) ? 'جميع الأنواع' : implode(', ', $allowedTypesArr);
        
        $text = Replies::get('INFO_BROADCAST_AWAITING_CONTENT');
        $text .= "\n\n📦 الحساب: {$account['name']}";
        $text .= "\n👥 عدد الزبائن: {$usersCount}";
        $text .= "\n📝 المحتوى المسموح: {$allowedTypesStr}";
        
        Telegram::sendMessage($chatId, $text);
    }
    
    /**
     * تنفيذ البث
     */
    public static function executeBroadcast(int $fromId, int $chatId, array $superdist): void
    {
        $data = State::getBroadcastData($fromId);
        if (!$data) {
            Telegram::sendMessage($chatId, "❌ لا توجد بيانات بث.");
            return;
        }
        
        State::clear($fromId);
        
        Telegram::sendMessage($chatId, Replies::get('INFO_BROADCAST_STARTED'));
        
        // تحديد الجمهور حسب نوع البث
        $audienceType = $data['audience_type'] ?? 'superdist_users';
        $audienceFilter = $data['audience_filter'] ?? null;
        
        if ($audienceType === 'account_users' && $audienceFilter) {
            // بث لزبائن حساب معين
            $filterData = json_decode($audienceFilter, true);
            $recipients = Db::fetchAll(
                "SELECT DISTINCT u.telegram_id
                 FROM users u
                 JOIN user_accounts ua ON ua.user_id = u.id
                 WHERE ua.superdist_id = ? AND ua.account_id = ? AND u.status = 'active'",
                [$filterData['superdist_id'], $filterData['account_id']]
            );
        } else {
            // بث لجميع زبائن السوبر موزع
            $recipients = Db::fetchAll(
                "SELECT DISTINCT u.telegram_id
                 FROM users u
                 JOIN user_accounts ua ON ua.user_id = u.id
                 WHERE ua.superdist_id = ? AND u.status = 'active'",
                [$superdist['id']]
            );
        }
        
        $success = 0;
        $failed = 0;
        
        // تحديد المحتوى الصحيح
        $contentType = $data['content_type'] ?? 'text';
        $content = ($contentType === 'text') 
            ? ($data['content_text'] ?? '') 
            : ($data['content_file_id'] ?? '');
        $caption = $data['content_caption'] ?? '';
        
        // التحقق من وجود محتوى
        if (empty($content)) {
            Telegram::sendMessage($chatId, "❌ المحتوى فارغ!");
            return;
        }
        
        foreach ($recipients as $r) {
            $telegramId = $r['telegram_id'] ?? null;
            if (!$telegramId) {
                $failed++;
                continue;
            }
            
            $result = Telegram::sendByType($telegramId, $contentType, $content, $caption);
            
            if ($result['ok'] ?? false) {
                $success++;
            } else {
                $failed++;
                Logger::error('Superdist broadcast failed', [
                    'telegram_id' => $telegramId,
                    'error' => $result['description'] ?? ($result['error'] ?? 'Unknown'),
                ]);
            }
            
            usleep(50000); // 50ms delay
        }
        
        // تسجيل البث
        Db::insert('broadcast_logs', [
            'broadcaster_type' => 'superdist',
            'broadcaster_id'   => (string)$superdist['id'],
            'audience_type'    => $audienceType,
            'audience_filter'  => $audienceFilter,
            'content_type'     => $data['content_type'] ?? 'text',
            'content_text'     => $data['content_text'] ?? null,
            'content_file_id'  => $data['content_file_id'] ?? null,
            'total_recipients' => count($recipients),
            'successful_sends' => $success,
            'failed_sends'     => $failed,
            'status'           => 'completed',
            'started_at'       => date('Y-m-d H:i:s'),
            'completed_at'     => date('Y-m-d H:i:s'),
        ]);
        
        Logger::broadcastCompleted('superdist', (string)$superdist['id'], $audienceType, count($recipients), $success, $failed);
        
        Telegram::sendMessage($chatId, Replies::get('INFO_BROADCAST_DONE', [
            'success' => $success,
            'failed'  => $failed,
            'total'   => count($recipients),
        ]));
    }
    
    // ================================================================
    // كود الاشتراك
    // ================================================================
    
    /**
     * معالجة قسيمة VCH-...
     * القسيمة تضيف حسابات للموزع (مثل SD- لكن بنظام Voucher)
     */
    public static function handleVoucherCode(string $code, int $fromId, int $chatId, array $superdist): void
    {
        $code = strtoupper(trim($code));
        
        // استخدام نظام Voucher للاسترداد
        $result = Voucher::redeem($code, $fromId);
        
        if (!$result['success']) {
            // رسائل خطأ محددة
            $errorMessages = [
                'القسيمة غير موجودة'         => "❌ <b>القسيمة غير موجودة</b>\n\nتأكد من كتابة الكود بشكل صحيح.",
                'القسيمة مستخدمة بالكامل'    => "⚠️ <b>القسيمة مستخدمة</b>\n\nهذه القسيمة تم استخدامها من قبل.",
                'القسيمة منتهية الصلاحية'    => "⏰ <b>القسيمة منتهية</b>\n\nانتهت صلاحية هذه القسيمة.",
                'القسيمة معطلة'              => "🚫 <b>القسيمة معطلة</b>\n\nهذه القسيمة تم تعطيلها من الإدارة.",
                'الموزع غير موجود'           => "❌ <b>خطأ</b>\n\nلم يتم التعرف عليك كموزع.",
            ];
            
            $errorText = $errorMessages[$result['error']] ?? "❌ {$result['error']}";
            Telegram::sendMessage($chatId, $errorText);
            return;
        }
        
        // النجاح - عرض النتيجة
        $data = $result['result'];
        $voucherType = $result['voucher_type'] ?? 'sd';
        
        if ($voucherType === 'sd') {
            // قسيمة حسابات
            $text = "✅ <b>تم تفعيل القسيمة بنجاح!</b>\n\n";
            $text .= "📦 <b>الحسابات المضافة:</b>\n";
            
            $accounts = $data['accounts_added'] ?? $data['accounts'] ?? [];
            foreach ($accounts as $acc) {
                $name = $acc['account_name'] ?? $acc['name'] ?? 'حساب';
                $days = $acc['duration_days'] ?? $acc['days'] ?? 0;
                $text .= "• {$name} ({$days} يوم)\n";
            }
            
            $text .= "\n💡 استخدم /my_accounts لعرض حساباتك.";
            
        } elseif ($voucherType === 'activation_bundle') {
            // حزمة أكواد تفعيل
            $text = "✅ <b>تم استلام حزمة الأكواد!</b>\n\n";
            $text .= "📦 <b>الحساب:</b> {$data['account_name']}\n";
            $text .= "🔢 <b>عدد الأكواد:</b> {$data['codes_count']}\n\n";
            $text .= "📋 <b>الأكواد:</b>\n";
            
            $codes = $data['codes'] ?? [];
            foreach (array_slice($codes, 0, 10) as $i => $c) {
                $text .= ($i + 1) . ". <code>{$c}</code>\n";
            }
            
            if (count($codes) > 10) {
                $text .= "... و " . (count($codes) - 10) . " أكواد أخرى\n";
            }
            
            $text .= "\n⚠️ احفظ هذه الأكواد، لن يتم عرضها مرة أخرى!";
        } else {
            $text = "✅ <b>تم تفعيل القسيمة بنجاح!</b>";
        }
        
        Telegram::sendLongMessage($chatId, $text);
        
        Logger::event('voucher_used_by_superdist', 'superdist', $superdist['id'], 'voucher', $code, [
            'voucher_type' => $voucherType,
        ]);
    }
    
    /**
     * معالجة كود اشتراك SD-...
     */
    public static function handleSubscriptionCode(string $code, int $fromId, int $chatId, array $superdist): void
    {
        $code = strtoupper(trim($code));
        
        // جلب الكود
        $subCode = Db::fetchOne(
            "SELECT * FROM superdist_subscription_codes WHERE code = ?",
            [$code]
        );
        
        if (!$subCode) {
            Telegram::sendMessage($chatId, Replies::get('ERROR_INVALID_CODE'));
            return;
        }
        
        if ($subCode['status'] !== 'unused') {
            if ($subCode['status'] === 'used') {
                Telegram::sendMessage($chatId, Replies::get('ERROR_CODE_USED'));
            } else {
                Telegram::sendMessage($chatId, Replies::get('ERROR_CODE_EXPIRED'));
            }
            return;
        }
        
        // التحقق من الاستخدامات
        if ($subCode['current_uses'] >= $subCode['max_uses']) {
            Telegram::sendMessage($chatId, Replies::get('ERROR_CODE_USED'));
            return;
        }
        
        // تطبيق الكود
        try {
            Db::beginTransaction();
            
            $accountsData = json_decode($subCode['accounts_data'], true);
            $addedAccounts = [];
            
            foreach ($accountsData as $accData) {
                $accountId = $accData['account_id'];
                $durationDays = $accData['duration_days'];
                
                // التحقق من وجود الربط
                $existing = Db::fetchOne(
                    "SELECT * FROM superdist_accounts WHERE superdist_id = ? AND account_id = ?",
                    [$superdist['id'], $accountId]
                );
                
                $startAt = date('Y-m-d H:i:s');
                $endAt = date('Y-m-d H:i:s', strtotime("+{$durationDays} days"));
                
                if ($existing) {
                    // تمديد الفترة
                    if (strtotime($existing['end_at']) > time()) {
                        // لا يزال نشطاً، أضف الأيام
                        $endAt = date('Y-m-d H:i:s', strtotime($existing['end_at'] . " +{$durationDays} days"));
                    }
                    
                    Db::update('superdist_accounts', [
                        'end_at'        => $endAt,
                        'duration_days' => $durationDays,
                    ], 'superdist_id = ? AND account_id = ?', [$superdist['id'], $accountId]);
                } else {
                    // إضافة جديدة
                    Db::insert('superdist_accounts', [
                        'superdist_id'  => $superdist['id'],
                        'account_id'    => $accountId,
                        'start_at'      => $startAt,
                        'duration_days' => $durationDays,
                        'end_at'        => $endAt,
                    ]);
                }
                
                $account = getAccountById($accountId);
                $addedAccounts[] = $account['name'] . " ({$durationDays} يوم)";
            }
            
            // تحديث الكود
            $newUses = $subCode['current_uses'] + 1;
            $newStatus = ($newUses >= $subCode['max_uses']) ? 'used' : 'unused';
            
            Db::update('superdist_subscription_codes', [
                'current_uses'          => $newUses,
                'status'                => $newStatus,
                'used_by_superdist_id'  => $superdist['id'],
                'used_at'               => date('Y-m-d H:i:s'),
            ], 'id = ?', [$subCode['id']]);
            
            Db::commit();
            
            Logger::event('superdist_subscription_activated', 'superdist', $superdist['id'], 'code', $code, [
                'accounts' => $addedAccounts,
            ]);
            
            $text = "✅ <b>تم تفعيل كود الاشتراك بنجاح!</b>\n\n";
            $text .= "الحسابات المضافة/المجددة:\n";
            foreach ($addedAccounts as $acc) {
                $text .= "• {$acc}\n";
            }
            $text .= "\nاستخدم /my_accounts لعرض حساباتك.";
            
            Telegram::sendMessage($chatId, $text);
            
        } catch (Throwable $e) {
            Db::rollback();
            Logger::error('Failed to activate subscription code', ['error' => $e->getMessage()]);
            Telegram::sendMessage($chatId, "❌ فشل تفعيل الكود. حاول مرة أخرى.");
        }
    }
    
    // ================================================================
    // دوال مساعدة
    // ================================================================
    
    /**
     * جلب ربط حساب نشط للسوبر موزع
     */
    private static function getActiveSuperdistAccount(int $superdistId, int $accountId): ?array
    {
        return Db::fetchOne(
            "SELECT sa.*, a.status as account_status
             FROM superdist_accounts sa
             JOIN accounts a ON a.id = sa.account_id
             WHERE sa.superdist_id = ? AND sa.account_id = ?
               AND sa.end_at > NOW() AND a.status = 'active'",
            [$superdistId, $accountId]
        );
    }
    
    /**
     * إرسال رسالة خطأ الحد
     * 
     * قيم reason من checkActivationGenerationLimit:
     *   - 'daily_limit', 'weekly_limit', 'monthly_limit'
     *   - 'no_access', 'expired', 'partial'
     */
    private static function sendLimitError(int $chatId, array $limitCheck, int $superdistId): void
    {
        $reason = $limitCheck['reason'] ?? 'daily_limit';
        
        $replyKey = match ($reason) {
            'daily_limit'             => 'ERROR_ACTIVATION_LIMIT_DAILY',
            'weekly_limit'            => 'ERROR_ACTIVATION_LIMIT_WEEKLY',
            'monthly_limit'           => 'ERROR_ACTIVATION_LIMIT_MONTHLY',
            'no_access'               => 'ERROR_ACCOUNT_NOT_ACTIVE',
            'expired'                 => 'ERROR_ACCOUNT_NOT_ACTIVE',
            'DAILY_LIMIT_EXCEEDED'    => 'ERROR_ACTIVATION_LIMIT_DAILY',
            'WEEKLY_LIMIT_EXCEEDED'   => 'ERROR_ACTIVATION_LIMIT_WEEKLY',
            'MONTHLY_LIMIT_EXCEEDED'  => 'ERROR_ACTIVATION_LIMIT_MONTHLY',
            default                   => 'ERROR_ACTIVATION_LIMIT_DAILY',
        };
        
        Telegram::sendMessage($chatId, Replies::getForSuperDist($replyKey, $superdistId, [
            'limit' => $limitCheck['limit'] ?? 0,
            'used'  => $limitCheck['used'] ?? 0,
        ]));
    }
    
    /**
     * فحص الحدود باستخدام سجل مقفل (FOR UPDATE)
     * 
     * يُستخدم داخل transaction بعد قفل سجل superdist_accounts
     * لضمان عدم وجود race condition
     * 
     * @param array $sa سجل superdist_accounts المقفل
     * @param int $requestedCount عدد الأكواد المطلوبة
     * @return array ['allowed' => bool, 'max_allowed' => int, 'reason' => string|null]
     */
    private static function checkLimitsWithLockedRecord(array $sa, int $requestedCount): array
    {
        $maxAllowed = $requestedCount;
        
        // التحقق من الحد اليومي
        if ($sa['activation_limit_daily'] !== null) {
            $remaining = $sa['activation_limit_daily'] - $sa['activations_today'];
            if ($remaining <= 0) {
                return [
                    'allowed'     => false,
                    'max_allowed' => 0,
                    'reason'      => 'daily_limit',
                    'limit'       => $sa['activation_limit_daily'],
                    'used'        => $sa['activations_today'],
                ];
            }
            $maxAllowed = min($maxAllowed, $remaining);
        }
        
        // التحقق من الحد الأسبوعي
        if ($sa['activation_limit_weekly'] !== null) {
            $remaining = $sa['activation_limit_weekly'] - $sa['activations_this_week'];
            if ($remaining <= 0) {
                return [
                    'allowed'     => false,
                    'max_allowed' => 0,
                    'reason'      => 'weekly_limit',
                    'limit'       => $sa['activation_limit_weekly'],
                    'used'        => $sa['activations_this_week'],
                ];
            }
            $maxAllowed = min($maxAllowed, $remaining);
        }
        
        // التحقق من الحد الشهري
        if ($sa['activation_limit_monthly'] !== null) {
            $remaining = $sa['activation_limit_monthly'] - $sa['activations_this_month'];
            if ($remaining <= 0) {
                return [
                    'allowed'     => false,
                    'max_allowed' => 0,
                    'reason'      => 'monthly_limit',
                    'limit'       => $sa['activation_limit_monthly'],
                    'used'        => $sa['activations_this_month'],
                ];
            }
            $maxAllowed = min($maxAllowed, $remaining);
        }
        
        // التحقق من أن المتبقي يكفي للطلب
        if ($maxAllowed < $requestedCount) {
            return [
                'allowed'     => false,
                'max_allowed' => $maxAllowed,
                'reason'      => 'partial',
            ];
        }
        
        return [
            'allowed'     => true,
            'max_allowed' => $maxAllowed,
            'reason'      => null,
        ];
    }
}
