<?php
/**
 * نظام فلترة البث (Broadcast Filter)
 * 
 * يتحكم في:
 * 1. صلاحيات أنواع المحتوى للموزعين
 * 2. فحص الكلمات المحظورة (Blacklist)
 * 3. تسجيل البثات المحظورة
 */

class BroadcastFilter
{
    // أنواع المحتوى المدعومة
    const CONTENT_TYPES = [
        'text',
        'stickers',
        'photos',
        'videos',
        'documents',
        'audio',
        'voice',
        'video_note',
        'animation',
    ];
    
    // الإعدادات المخزنة مؤقتاً
    private static ?array $settings = null;
    private static ?array $blacklist = null;
    
    // ================================================================
    // صلاحيات أنواع المحتوى
    // ================================================================
    
    /**
     * التحقق من صلاحية نوع المحتوى للموزع
     * 
     * @param int $superdistId معرف الموزع
     * @param string $contentType نوع المحتوى
     * @return array ['allowed' => bool, 'reason' => string]
     */
    public static function checkContentTypePermission(int $superdistId, string $contentType): array
    {
        // جلب بيانات الموزع
        $superdist = Db::fetchOne(
            "SELECT can_broadcast, broadcast_allowed_types FROM super_distributors WHERE id = ?",
            [$superdistId]
        );
        
        if (!$superdist) {
            return ['allowed' => false, 'reason' => 'الموزع غير موجود'];
        }
        
        // التحقق من تفعيل البث
        if (!$superdist['can_broadcast']) {
            return ['allowed' => false, 'reason' => 'البث معطل لحسابك'];
        }
        
        // التحقق من نوع المحتوى المسموح
        $allowedTypes = json_decode($superdist['broadcast_allowed_types'] ?? '[]', true);
        
        // إذا لم تُحدد أنواع، السماح بالكل
        if (empty($allowedTypes)) {
            return ['allowed' => true, 'reason' => 'OK'];
        }
        
        // تطبيع نوع المحتوى
        $normalizedType = self::normalizeContentType($contentType);
        
        if (!in_array($normalizedType, $allowedTypes)) {
            $allowedList = implode(', ', $allowedTypes);
            return [
                'allowed' => false,
                'reason' => "نوع المحتوى '{$contentType}' غير مسموح. الأنواع المسموحة: {$allowedList}",
                'allowed_types' => $allowedTypes,
            ];
        }
        
        return ['allowed' => true, 'reason' => 'OK'];
    }
    
    /**
     * تطبيع نوع المحتوى
     */
    private static function normalizeContentType(string $type): string
    {
        $map = [
            'photo' => 'photos',
            'video' => 'videos',
            'document' => 'documents',
            'sticker' => 'stickers',
            'voice' => 'voice',
            'video_note' => 'video_note',
            'animation' => 'animation',
            'audio' => 'audio',
        ];
        
        return $map[$type] ?? $type;
    }
    
    /**
     * تعيين صلاحيات البث للموزع
     */
    public static function setSuperdistBroadcastPermissions(int $superdistId, bool $canBroadcast, array $allowedTypes = []): bool
    {
        // التحقق من صحة الأنواع
        $validTypes = array_intersect($allowedTypes, self::CONTENT_TYPES);
        
        return Db::update('super_distributors', [
            'can_broadcast' => $canBroadcast ? 1 : 0,
            'broadcast_allowed_types' => json_encode($validTypes),
        ], 'id = ?', [$superdistId]) > 0;
    }
    
    /**
     * جلب صلاحيات البث للموزع
     */
    public static function getSuperdistBroadcastPermissions(int $superdistId): array
    {
        $superdist = Db::fetchOne(
            "SELECT can_broadcast, broadcast_allowed_types FROM super_distributors WHERE id = ?",
            [$superdistId]
        );
        
        if (!$superdist) {
            return ['can_broadcast' => false, 'allowed_types' => []];
        }
        
        return [
            'can_broadcast' => (bool)$superdist['can_broadcast'],
            'allowed_types' => json_decode($superdist['broadcast_allowed_types'] ?? '[]', true),
        ];
    }
    
    // ================================================================
    // نظام القائمة السوداء (Blacklist)
    // ================================================================
    
    /**
     * فحص المحتوى ضد القائمة السوداء
     * 
     * @param string $content المحتوى (نص أو caption)
     * @param string|null $filename اسم الملف (اختياري)
     * @return array ['clean' => bool, 'matches' => array, 'severity' => string]
     */
    public static function checkBlacklist(string $content, ?string $filename = null): array
    {
        // التحقق من تفعيل النظام
        if (!self::getSetting('blacklist_enabled', true)) {
            return ['clean' => true, 'matches' => [], 'severity' => null];
        }
        
        // جلب القائمة السوداء
        $blacklist = self::getBlacklist();
        
        if (empty($blacklist)) {
            return ['clean' => true, 'matches' => [], 'severity' => null];
        }
        
        $matches = [];
        $highestSeverity = null;
        
        // دمج المحتوى مع اسم الملف
        $fullContent = strtolower($content . ' ' . ($filename ?? ''));
        
        foreach ($blacklist as $item) {
            $matched = false;
            
            switch ($item['type']) {
                case 'word':
                    // مطابقة كلمة (case-insensitive)
                    if (stripos($fullContent, strtolower($item['word'])) !== false) {
                        $matched = true;
                    }
                    break;
                    
                case 'regex':
                    // مطابقة regex
                    if (preg_match('/' . $item['word'] . '/i', $fullContent)) {
                        $matched = true;
                    }
                    break;
                    
                case 'domain':
                    // مطابقة نطاق/رابط
                    if (stripos($fullContent, strtolower($item['word'])) !== false) {
                        $matched = true;
                    }
                    break;
            }
            
            if ($matched) {
                $matches[] = [
                    'word' => $item['word'],
                    'type' => $item['type'],
                    'severity' => $item['severity'],
                ];
                
                // تحديد أعلى درجة خطورة
                if ($highestSeverity === null || 
                    self::severityLevel($item['severity']) > self::severityLevel($highestSeverity)) {
                    $highestSeverity = $item['severity'];
                }
            }
        }
        
        return [
            'clean' => empty($matches),
            'matches' => $matches,
            'severity' => $highestSeverity,
        ];
    }
    
    /**
     * تحويل الخطورة إلى رقم للمقارنة
     */
    private static function severityLevel(string $severity): int
    {
        return match($severity) {
            'block' => 3,
            'warn' => 2,
            'log' => 1,
            default => 0,
        };
    }
    
    /**
     * جلب القائمة السوداء
     */
    public static function getBlacklist(): array
    {
        if (self::$blacklist === null) {
            self::$blacklist = Db::fetchAll(
                "SELECT word, type, severity FROM broadcast_blacklist WHERE is_active = 1"
            ) ?? [];
        }
        return self::$blacklist;
    }
    
    /**
     * إضافة كلمة للقائمة السوداء
     */
    public static function addToBlacklist(string $word, string $type = 'word', string $severity = 'block', ?int $adminId = null): bool
    {
        try {
            Db::insert('broadcast_blacklist', [
                'word' => $word,
                'type' => $type,
                'severity' => $severity,
                'is_active' => 1,
                'added_by_admin_id' => $adminId,
            ]);
            
            // مسح الكاش
            self::$blacklist = null;
            
            return true;
        } catch (Throwable $e) {
            return false;
        }
    }
    
    /**
     * حذف كلمة من القائمة السوداء
     */
    public static function removeFromBlacklist(int $id): bool
    {
        $result = Db::delete('broadcast_blacklist', 'id = ?', [$id]) > 0;
        self::$blacklist = null;
        return $result;
    }
    
    /**
     * تفعيل/تعطيل كلمة
     */
    public static function toggleBlacklistItem(int $id, bool $active): bool
    {
        $result = Db::update('broadcast_blacklist', ['is_active' => $active ? 1 : 0], 'id = ?', [$id]) > 0;
        self::$blacklist = null;
        return $result;
    }
    
    /**
     * جلب جميع عناصر القائمة السوداء (للإدارة)
     */
    public static function getAllBlacklistItems(): array
    {
        return Db::fetchAll("SELECT * FROM broadcast_blacklist ORDER BY created_at DESC") ?? [];
    }
    
    // ================================================================
    // تسجيل البثات المحظورة
    // ================================================================
    
    /**
     * تسجيل بث محظور
     */
    public static function logBlockedBroadcast(
        string $broadcasterType,
        int $broadcasterId,
        int $broadcasterTelegramId,
        string $contentType,
        string $contentSnippet,
        array $matchedWords,
        string $reason,
        string $actionTaken = 'blocked'
    ): void {
        try {
            Db::insert('blocked_broadcasts', [
                'broadcaster_type' => $broadcasterType,
                'broadcaster_id' => $broadcasterId,
                'broadcaster_telegram_id' => $broadcasterTelegramId,
                'content_type' => $contentType,
                'content_snippet' => mb_substr($contentSnippet, 0, 500),
                'matched_words' => json_encode($matchedWords, JSON_UNESCAPED_UNICODE),
                'reason' => $reason,
                'action_taken' => $actionTaken,
            ]);
            
            Logger::event('broadcast_blocked', $broadcasterType, $broadcasterId, 'reason', $reason, [
                'content_type' => $contentType,
                'matched_words' => $matchedWords,
            ]);
        } catch (Throwable $e) {
            Logger::error('Failed to log blocked broadcast', ['error' => $e->getMessage()]);
        }
    }
    
    /**
     * جلب البثات المحظورة
     */
    public static function getBlockedBroadcasts(int $limit = 50, int $offset = 0): array
    {
        return Db::fetchAll(
            "SELECT * FROM blocked_broadcasts ORDER BY created_at DESC LIMIT ? OFFSET ?",
            [$limit, $offset]
        ) ?? [];
    }
    
    // ================================================================
    // إعدادات البث
    // ================================================================
    
    /**
     * جلب إعداد
     */
    public static function getSetting(string $key, $default = null)
    {
        if (self::$settings === null) {
            self::loadSettings();
        }
        
        return self::$settings[$key] ?? $default;
    }
    
    /**
     * تعيين إعداد
     */
    public static function setSetting(string $key, string $value): bool
    {
        try {
            $exists = Db::exists('broadcast_settings', 'setting_key = ?', [$key]);
            
            if ($exists) {
                Db::update('broadcast_settings', ['setting_value' => $value], 'setting_key = ?', [$key]);
            } else {
                Db::insert('broadcast_settings', [
                    'setting_key' => $key,
                    'setting_value' => $value,
                ]);
            }
            
            // تحديث الكاش
            self::$settings[$key] = $value;
            
            return true;
        } catch (Throwable $e) {
            return false;
        }
    }
    
    /**
     * تحميل الإعدادات
     */
    private static function loadSettings(): void
    {
        self::$settings = [];
        
        try {
            $rows = Db::fetchAll("SELECT setting_key, setting_value FROM broadcast_settings");
            foreach ($rows ?? [] as $row) {
                self::$settings[$row['setting_key']] = $row['setting_value'];
            }
        } catch (Throwable $e) {
            // الجدول قد لا يكون موجوداً بعد
        }
    }
    
    /**
     * جلب جميع الإعدادات
     */
    public static function getAllSettings(): array
    {
        if (self::$settings === null) {
            self::loadSettings();
        }
        return self::$settings;
    }
    
    // ================================================================
    // فحص شامل قبل البث
    // ================================================================
    
    /**
     * فحص شامل للمحتوى قبل البث
     * 
     * @param string $broadcasterType 'admin' أو 'superdist'
     * @param int $broadcasterId معرف المُرسل
     * @param int $broadcasterTelegramId Telegram ID
     * @param string $contentType نوع المحتوى
     * @param string $content المحتوى النصي
     * @param string|null $filename اسم الملف
     * @return array ['allowed' => bool, 'reason' => string, 'action' => string]
     */
    public static function checkBroadcast(
        string $broadcasterType,
        int $broadcasterId,
        int $broadcasterTelegramId,
        string $contentType,
        string $content,
        ?string $filename = null
    ): array {
        // Admin يتجاوز فحص الصلاحيات (ما لم يُفعّل الإعداد)
        $adminBypassBlacklist = (bool)self::getSetting('admin_bypass_blacklist', false);
        
        if ($broadcasterType === 'admin' && $adminBypassBlacklist) {
            return ['allowed' => true, 'reason' => 'Admin bypass', 'action' => 'allow'];
        }
        
        // فحص صلاحية نوع المحتوى للموزع
        if ($broadcasterType === 'superdist') {
            $permCheck = self::checkContentTypePermission($broadcasterId, $contentType);
            if (!$permCheck['allowed']) {
                self::logBlockedBroadcast(
                    $broadcasterType, $broadcasterId, $broadcasterTelegramId,
                    $contentType, $content, [], $permCheck['reason'], 'blocked'
                );
                return [
                    'allowed' => false,
                    'reason' => $permCheck['reason'],
                    'action' => 'blocked',
                ];
            }
        }
        
        // فحص القائمة السوداء
        if (self::getSetting('moderation_enabled', true)) {
            $blacklistCheck = self::checkBlacklist($content, $filename);
            
            if (!$blacklistCheck['clean']) {
                $matchedWords = array_column($blacklistCheck['matches'], 'word');
                $severity = $blacklistCheck['severity'];
                
                $action = match($severity) {
                    'block' => 'blocked',
                    'warn' => 'warned',
                    'log' => 'logged',
                    default => 'blocked',
                };
                
                self::logBlockedBroadcast(
                    $broadcasterType, $broadcasterId, $broadcasterTelegramId,
                    $contentType, $content, $matchedWords,
                    "مطابقة كلمات محظورة: " . implode(', ', $matchedWords),
                    $action
                );
                
                if ($action === 'blocked') {
                    return [
                        'allowed' => false,
                        'reason' => '❌ تم رفض البث بسبب محتوى محظور: ' . implode(', ', $matchedWords),
                        'action' => 'blocked',
                        'matched_words' => $matchedWords,
                    ];
                } elseif ($action === 'warned') {
                    return [
                        'allowed' => true,
                        'reason' => '⚠️ تحذير: المحتوى يحتوي على كلمات مشبوهة',
                        'action' => 'warned',
                        'matched_words' => $matchedWords,
                    ];
                }
            }
        }
        
        return ['allowed' => true, 'reason' => 'OK', 'action' => 'allow'];
    }
}
