<?php
/**
 * CLI Script: معالجة الأكواد
 * 
 * يستخدم لمعالجة أكواد التفعيل بناءً على نافذة زمنية أو عدد محدد
 * 
 * الاستخدام:
 *   php cli/process_codes.php <mode> <value> <action> [options]
 * 
 * الأنماط (mode):
 *   since   - منذ تاريخ معين (YYYY-MM-DD أو Nd للأيام مثل 30d)
 *   last_n  - آخر N كود
 * 
 * الإجراءات (action):
 *   expire_unused   - تحويل الأكواد غير المستخدمة إلى منتهية
 *   disable_unused  - تعطيل الأكواد غير المستخدمة
 *   report_only     - إنشاء تقرير فقط بدون تغيير
 * 
 * الخيارات:
 *   --account=<slug>    - فلتر حسب الحساب
 *   --superdist=<id>    - فلتر حسب الموزع
 *   --run=cron          - وضع الـ cron (بدون تأكيد)
 * 
 * أمثلة:
 *   php cli/process_codes.php since 30d report_only
 *   php cli/process_codes.php since 2025-11-01 expire_unused
 *   php cli/process_codes.php last_n 500 disable_unused --account=vip-acc
 *   php cli/process_codes.php since 60d expire_unused --run=cron
 */

// التأكد من تشغيل CLI
if (php_sapi_name() !== 'cli') {
    die("هذا الملف يعمل من سطر الأوامر فقط.\n");
}

// تحميل النظام
define('BOT_TYPE', 'admin');
require_once __DIR__ . '/../libs/bootstrap.php';

// ============================================================
// معالجة المعاملات
// ============================================================

$args = $argv;
array_shift($args); // إزالة اسم الملف

if (count($args) < 3) {
    printUsage();
    exit(1);
}

$mode = strtolower($args[0]);
$value = $args[1];
$action = strtolower($args[2]);

// تحليل الخيارات
$options = parseOptions(array_slice($args, 3));
$accountSlug = $options['account'] ?? null;
$superdistId = isset($options['superdist']) ? (int)$options['superdist'] : null;
$isCron = ($options['run'] ?? '') === 'cron';

// التحقق من صحة المعاملات
if (!in_array($mode, ['since', 'last_n'])) {
    echo "❌ خطأ: النمط غير صالح: {$mode}\n";
    echo "الأنماط المتاحة: since, last_n\n";
    exit(1);
}

if (!in_array($action, ['expire_unused', 'disable_unused', 'report_only'])) {
    echo "❌ خطأ: الإجراء غير صالح: {$action}\n";
    echo "الإجراءات المتاحة: expire_unused, disable_unused, report_only\n";
    exit(1);
}

// جلب معرف الحساب إذا تم تحديده
$accountId = null;
if ($accountSlug) {
    $account = getAccountBySlug($accountSlug);
    if (!$account) {
        echo "❌ خطأ: الحساب غير موجود: {$accountSlug}\n";
        exit(1);
    }
    $accountId = $account['id'];
}

// ============================================================
// عرض معلومات المهمة
// ============================================================

echo "\n";
echo "═══════════════════════════════════════════════════════════\n";
echo "           معالجة الأكواد - Process Codes Window           \n";
echo "═══════════════════════════════════════════════════════════\n\n";

echo "📋 معلومات المهمة:\n";
echo "   • النمط: {$mode}\n";
echo "   • القيمة: {$value}\n";
echo "   • الإجراء: {$action}\n";
if ($accountSlug) echo "   • الحساب: {$accountSlug}\n";
if ($superdistId) echo "   • الموزع: #{$superdistId}\n";
echo "\n";

// طلب التأكيد (إلا في وضع cron)
if (!$isCron && $action !== 'report_only') {
    echo "⚠️  هل أنت متأكد من تنفيذ هذا الإجراء؟\n";
    echo "    اكتب 'yes' للمتابعة: ";
    
    $handle = fopen("php://stdin", "r");
    $line = trim(fgets($handle));
    fclose($handle);
    
    if (strtolower($line) !== 'yes') {
        echo "\n❌ تم الإلغاء.\n";
        exit(0);
    }
    echo "\n";
}

// ============================================================
// تنفيذ المهمة
// ============================================================

echo "⏳ جاري التنفيذ...\n\n";

$startTime = microtime(true);

try {
    $result = ProcessCodesWindow::run(
        $mode,
        $value,
        $action,
        $accountId,
        $superdistId,
        0 // CLI initiated
    );
    
    $duration = round((microtime(true) - $startTime) * 1000);
    
    if ($result['success']) {
        echo "✅ اكتملت المعالجة بنجاح!\n\n";
        echo "📊 النتائج:\n";
        echo "   • الإجمالي: {$result['total']}\n";
        echo "   • تمت معالجتها: {$result['processed']}\n";
        echo "   • تأثرت: {$result['affected']}\n";
        echo "   • تم تخطيها: {$result['skipped']}\n";
        echo "   • المدة: {$duration}ms\n\n";
        
        echo "📁 الملفات:\n";
        echo "   • التقرير: {$result['report_path']}\n";
        echo "   • CSV: {$result['csv_path']}\n\n";
        
        echo "🔗 Job ID: {$result['job_id']}\n";
        
        exit(0);
    } else {
        echo "❌ فشلت المعالجة:\n";
        echo "   {$result['error']}\n\n";
        exit(1);
    }
    
} catch (Exception $e) {
    echo "❌ خطأ غير متوقع:\n";
    echo "   {$e->getMessage()}\n\n";
    exit(1);
}

// ============================================================
// الدوال المساعدة
// ============================================================

function printUsage(): void
{
    echo "\n";
    echo "═══════════════════════════════════════════════════════════\n";
    echo "           معالجة الأكواد - Process Codes Window           \n";
    echo "═══════════════════════════════════════════════════════════\n\n";
    
    echo "الاستخدام:\n";
    echo "  php cli/process_codes.php <mode> <value> <action> [options]\n\n";
    
    echo "الأنماط (mode):\n";
    echo "  since     منذ تاريخ معين (YYYY-MM-DD أو Nd مثل 30d)\n";
    echo "  last_n    آخر N كود\n\n";
    
    echo "الإجراءات (action):\n";
    echo "  expire_unused    تحويل الأكواد غير المستخدمة إلى منتهية\n";
    echo "  disable_unused   تعطيل الأكواد غير المستخدمة\n";
    echo "  report_only      إنشاء تقرير فقط بدون تغيير\n\n";
    
    echo "الخيارات:\n";
    echo "  --account=<slug>    فلتر حسب الحساب\n";
    echo "  --superdist=<id>    فلتر حسب الموزع\n";
    echo "  --run=cron          وضع الـ cron (بدون تأكيد)\n\n";
    
    echo "أمثلة:\n";
    echo "  php cli/process_codes.php since 30d report_only\n";
    echo "  php cli/process_codes.php since 2025-11-01 expire_unused\n";
    echo "  php cli/process_codes.php last_n 500 disable_unused --account=vip\n";
    echo "  php cli/process_codes.php since 60d expire_unused --run=cron\n\n";
    
    echo "للتشغيل التلقائي (Cron) شهرياً:\n";
    echo "  0 3 1 * * php /path/to/cli/process_codes.php since 30d expire_unused --run=cron\n\n";
}

function parseOptions(array $args): array
{
    $options = [];
    foreach ($args as $arg) {
        if (strpos($arg, '--') === 0) {
            $arg = substr($arg, 2);
            if (strpos($arg, '=') !== false) {
                [$key, $value] = explode('=', $arg, 2);
                $options[$key] = $value;
            } else {
                $options[$arg] = true;
            }
        }
    }
    return $options;
}
