<?php
/**
 * API v1 Entry Point
 * 
 * نقطة دخول واجهة API
 * 
 * المسارات المدعومة:
 * - POST /api/v1/vouchers - إنشاء قسيمة
 * - GET /api/v1/vouchers/{code} - جلب قسيمة
 * - POST /api/v1/vouchers/redeem - استرداد قسيمة
 * - POST /api/v1/payments/webhook - webhook الدفع
 * - POST /api/v1/system/check - تشغيل فحص النظام
 * - GET /api/v1/system/reports - جلب تقارير الفحص
 */

// منع الوصول المباشر
if (php_sapi_name() === 'cli') {
    die("This script cannot be run from CLI\n");
}

// تعريف نوع البوت
define('BOT_TYPE', 'api');

// تحميل الملفات الأساسية
require __DIR__ . '/../../libs/bootstrap.php';

// إعداد الـ headers
header('Content-Type: application/json; charset=utf-8');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');

// التعامل مع CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization, X-API-Key');
    http_response_code(204);
    exit;
}

// ================================================================
// Router بسيط
// ================================================================

class ApiRouter
{
    private static array $routes = [];
    
    public static function post(string $path, callable $handler): void
    {
        self::$routes['POST'][$path] = $handler;
    }
    
    public static function get(string $path, callable $handler): void
    {
        self::$routes['GET'][$path] = $handler;
    }
    
    public static function run(): void
    {
        $method = $_SERVER['REQUEST_METHOD'];
        $uri = $_SERVER['REQUEST_URI'];
        
        // إزالة query string
        $uri = strtok($uri, '?');
        
        // إزالة /api/v1 من البداية
        $uri = preg_replace('#^/api/v1#', '', $uri);
        $uri = rtrim($uri, '/') ?: '/';
        
        // البحث عن مسار مطابق
        $routes = self::$routes[$method] ?? [];
        
        foreach ($routes as $pattern => $handler) {
            // تحويل المسار إلى regex
            $regex = preg_replace('#\{(\w+)\}#', '(?P<$1>[^/]+)', $pattern);
            $regex = '#^' . $regex . '$#';
            
            if (preg_match($regex, $uri, $matches)) {
                // استخراج المعاملات
                $params = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);
                
                try {
                    $result = $handler($params);
                    self::json($result);
                } catch (ApiException $e) {
                    self::error($e->getMessage(), $e->getCode());
                } catch (Throwable $e) {
                    Logger::error('API Error', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
                    self::error('Internal server error', 500);
                }
                return;
            }
        }
        
        self::error('Not found', 404);
    }
    
    public static function json($data, int $code = 200): void
    {
        http_response_code($code);
        echo json_encode([
            'success' => $code >= 200 && $code < 300,
            'data' => $data,
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }
    
    public static function error(string $message, int $code = 400): void
    {
        http_response_code($code);
        echo json_encode([
            'success' => false,
            'error' => $message,
        ], JSON_UNESCAPED_UNICODE);
    }
}

class ApiException extends Exception {}

// ================================================================
// Middleware للمصادقة
// ================================================================

function requireAdminAuth(): void
{
    $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? '';
    $configApiKey = config('admin_api_key');
    
    if (empty($configApiKey) || $apiKey !== $configApiKey) {
        throw new ApiException('Unauthorized', 401);
    }
}

function requireWebhookSignature(string $provider): void
{
    $signature = $_SERVER['HTTP_X_WEBHOOK_SIGNATURE'] ?? '';
    $secret = config("payment_{$provider}_webhook_secret");
    
    if (empty($secret)) {
        throw new ApiException('Webhook secret not configured', 500);
    }
    
    $payload = file_get_contents('php://input');
    $expectedSignature = hash_hmac('sha256', $payload, $secret);
    
    if (!hash_equals($expectedSignature, $signature)) {
        throw new ApiException('Invalid signature', 401);
    }
}

function getJsonBody(): array
{
    $body = file_get_contents('php://input');
    $data = json_decode($body, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new ApiException('Invalid JSON body', 400);
    }
    
    return $data ?? [];
}

// ================================================================
// تسجيل المسارات
// ================================================================

// ---------------------------------------------------------------
// القسائم (Vouchers)
// ---------------------------------------------------------------

// إنشاء قسيمة جديدة
ApiRouter::post('/vouchers', function($params) {
    requireAdminAuth();
    
    $data = getJsonBody();
    
    $type = $data['type'] ?? '';
    $payload = $data['payload'] ?? [];
    $options = [
        'max_uses' => $data['max_uses'] ?? 1,
        'expires_at' => $data['expires_at'] ?? null,
        'created_by_admin_id' => $data['admin_id'] ?? null,
    ];
    
    $result = Voucher::create($type, $payload, $options);
    
    if (!$result['success']) {
        throw new ApiException($result['error'], 400);
    }
    
    return [
        'voucher_id' => $result['voucher_id'],
        'code' => $result['code'],
    ];
});

// جلب قسيمة بالكود
ApiRouter::get('/vouchers/{code}', function($params) {
    requireAdminAuth();
    
    $voucher = Voucher::getByCode($params['code']);
    
    if (!$voucher) {
        throw new ApiException('Voucher not found', 404);
    }
    
    // فك JSON
    $voucher['payload'] = json_decode($voucher['payload'], true);
    
    return $voucher;
});

// استرداد قسيمة
ApiRouter::post('/vouchers/redeem', function($params) {
    $data = getJsonBody();
    
    $code = $data['code'] ?? '';
    $telegramId = $data['redeemer_telegram_id'] ?? 0;
    
    if (empty($code) || empty($telegramId)) {
        throw new ApiException('Missing required fields: code, redeemer_telegram_id', 400);
    }
    
    $result = Voucher::redeem($code, $telegramId);
    
    if (!$result['success']) {
        throw new ApiException($result['error'], 400);
    }
    
    return $result['result'];
});

// قائمة القسائم
ApiRouter::get('/vouchers', function($params) {
    requireAdminAuth();
    
    $filters = [
        'type' => $_GET['type'] ?? null,
        'status' => $_GET['status'] ?? null,
    ];
    
    $limit = min((int)($_GET['limit'] ?? 50), 100);
    $offset = (int)($_GET['offset'] ?? 0);
    
    $vouchers = Voucher::getList(array_filter($filters), $limit, $offset);
    
    // فك JSON
    foreach ($vouchers as &$v) {
        $v['payload'] = json_decode($v['payload'], true);
    }
    
    return $vouchers;
});

// ---------------------------------------------------------------
// الدفع (Payments)
// ---------------------------------------------------------------

// Webhook الدفع
ApiRouter::post('/payments/webhook', function($params) {
    $data = getJsonBody();
    
    $provider = $data['provider'] ?? 'generic';
    
    // التحقق من التوقيع (اختياري حسب المزود)
    // requireWebhookSignature($provider);
    
    // تسجيل الـ webhook
    try {
        Db::insert('payment_logs', [
            'provider' => $provider,
            'payment_id' => $data['payment_id'] ?? uniqid(),
            'event_type' => $data['event_type'] ?? 'payment.completed',
            'amount' => $data['amount'] ?? 0,
            'currency' => $data['currency'] ?? 'USD',
            'payer_email' => $data['payer_email'] ?? null,
            'payer_name' => $data['payer_name'] ?? null,
            'raw_payload' => json_encode($data, JSON_UNESCAPED_UNICODE),
            'signature_valid' => 1,
        ]);
    } catch (Throwable $e) {
        Logger::error('Failed to log payment webhook', ['error' => $e->getMessage()]);
    }
    
    // إنشاء قسيمة من الدفع
    if (($data['event_type'] ?? '') === 'payment.completed' && !empty($data['metadata'])) {
        $voucherResult = Voucher::createFromPayment([
            'payment_id' => $data['payment_id'],
            'provider' => $provider,
            'amount' => $data['amount'],
            'currency' => $data['currency'],
            'metadata' => $data['metadata'],
        ]);
        
        if ($voucherResult['success']) {
            // إرسال إشعار للمستفيد
            $telegramId = $data['metadata']['telegram_id'] ?? null;
            if ($telegramId) {
                try {
                    Telegram::sendMessage($telegramId, 
                        "✅ تم تأكيد الدفع بنجاح!\n\n" .
                        "🎟️ كود القسيمة: <code>{$voucherResult['code']}</code>\n\n" .
                        "استخدم الكود لتفعيل اشتراكك."
                    );
                } catch (Throwable $e) {
                    Logger::error('Failed to notify user about payment', ['error' => $e->getMessage()]);
                }
            }
            
            return [
                'status' => 'voucher_created',
                'voucher_code' => $voucherResult['code'],
            ];
        }
    }
    
    return ['status' => 'received'];
});

// ---------------------------------------------------------------
// فحص النظام (System Check)
// ---------------------------------------------------------------

// تشغيل فحص النظام
ApiRouter::post('/system/check', function($params) {
    requireAdminAuth();
    
    $data = getJsonBody();
    $type = $data['type'] ?? 'full';
    $adminId = $data['admin_id'] ?? null;
    
    $report = match($type) {
        'quick' => SystemChecker::runQuick($adminId, 'api'),
        'custom' => SystemChecker::runCustom($data['tests'] ?? [], $adminId, 'api'),
        default => SystemChecker::runFull($adminId, 'api'),
    };
    
    return [
        'report_id' => $report['report_id'] ?? null,
        'stats' => $report['stats'],
        'summary' => $report['summary'],
        'report_file_json' => $report['report_file_json'] ?? null,
    ];
});

// جلب تقارير الفحص
ApiRouter::get('/system/reports', function($params) {
    requireAdminAuth();
    
    $limit = min((int)($_GET['limit'] ?? 20), 50);
    $reports = SystemChecker::getReports($limit);
    
    // فك JSON
    foreach ($reports as &$r) {
        $r['results'] = json_decode($r['results'], true);
        $r['recommendations'] = json_decode($r['recommendations'], true);
    }
    
    return $reports;
});

// جلب تقرير محدد
ApiRouter::get('/system/reports/{id}', function($params) {
    requireAdminAuth();
    
    $report = SystemChecker::getReport((int)$params['id']);
    
    if (!$report) {
        throw new ApiException('Report not found', 404);
    }
    
    $report['results'] = json_decode($report['results'], true);
    $report['recommendations'] = json_decode($report['recommendations'], true);
    
    return $report;
});

// ---------------------------------------------------------------
// إعدادات البث
// ---------------------------------------------------------------

// جلب إعدادات البث
ApiRouter::get('/broadcast/settings', function($params) {
    requireAdminAuth();
    return BroadcastFilter::getAllSettings();
});

// تحديث إعداد
ApiRouter::post('/broadcast/settings', function($params) {
    requireAdminAuth();
    
    $data = getJsonBody();
    
    foreach ($data as $key => $value) {
        BroadcastFilter::setSetting($key, $value);
    }
    
    return BroadcastFilter::getAllSettings();
});

// جلب القائمة السوداء
ApiRouter::get('/broadcast/blacklist', function($params) {
    requireAdminAuth();
    return BroadcastFilter::getAllBlacklistItems();
});

// إضافة للقائمة السوداء
ApiRouter::post('/broadcast/blacklist', function($params) {
    requireAdminAuth();
    
    $data = getJsonBody();
    
    $word = $data['word'] ?? '';
    $type = $data['type'] ?? 'word';
    $severity = $data['severity'] ?? 'block';
    $adminId = $data['admin_id'] ?? null;
    
    if (empty($word)) {
        throw new ApiException('Word is required', 400);
    }
    
    $success = BroadcastFilter::addToBlacklist($word, $type, $severity, $adminId);
    
    if (!$success) {
        throw new ApiException('Failed to add word', 500);
    }
    
    return ['success' => true];
});

// ---------------------------------------------------------------
// البثات المحظورة
// ---------------------------------------------------------------

ApiRouter::get('/broadcast/blocked', function($params) {
    requireAdminAuth();
    
    $limit = min((int)($_GET['limit'] ?? 50), 100);
    $offset = (int)($_GET['offset'] ?? 0);
    
    return BroadcastFilter::getBlockedBroadcasts($limit, $offset);
});

// ---------------------------------------------------------------
// الحسابات (Accounts) - إضافة دفعة
// ---------------------------------------------------------------

ApiRouter::post('/accounts/batch', function($params) {
    requireAdminAuth();
    
    $data = getJsonBody();
    $accounts = $data['accounts'] ?? [];
    $mode = $data['mode'] ?? 'fail-fast'; // fail-fast أو partial-commit
    $preview = $data['preview'] ?? false;
    
    if (empty($accounts)) {
        throw new ApiException('No accounts provided', 400);
    }
    
    $results = [
        'total' => count($accounts),
        'success' => 0,
        'failed' => 0,
        'errors' => [],
        'created' => [],
    ];
    
    if ($preview) {
        // معاينة فقط
        foreach ($accounts as $i => $acc) {
            $validation = validateAccountData($acc);
            if (!$validation['valid']) {
                $results['errors'][] = ['index' => $i, 'data' => $acc, 'error' => $validation['error']];
                $results['failed']++;
            } else {
                $results['created'][] = ['index' => $i, 'data' => $acc, 'status' => 'will_create'];
                $results['success']++;
            }
        }
        return $results;
    }
    
    // التنفيذ الفعلي
    if ($mode === 'fail-fast') {
        Db::beginTransaction();
    }
    
    try {
        foreach ($accounts as $i => $acc) {
            $validation = validateAccountData($acc);
            if (!$validation['valid']) {
                $results['errors'][] = ['index' => $i, 'data' => $acc, 'error' => $validation['error']];
                $results['failed']++;
                
                if ($mode === 'fail-fast') {
                    Db::rollback();
                    throw new ApiException("Validation failed at index {$i}: {$validation['error']}", 400);
                }
                continue;
            }
            
            // إدراج الحساب
            Db::insert('accounts', [
                'name' => $acc['name'],
                'slug' => $acc['slug'],
                'secret_key' => $acc['secret_key'] ?? bin2hex(random_bytes(16)),
                'default_access_type' => $acc['access_type'] ?? 'lifetime',
                'default_access_days' => $acc['access_days'] ?? null,
                'status' => 'active',
            ]);
            
            $results['created'][] = ['index' => $i, 'id' => Db::lastInsertId(), 'slug' => $acc['slug']];
            $results['success']++;
        }
        
        if ($mode === 'fail-fast') {
            Db::commit();
        }
        
    } catch (ApiException $e) {
        throw $e;
    } catch (Throwable $e) {
        if ($mode === 'fail-fast') {
            Db::rollback();
        }
        throw new ApiException('Database error: ' . $e->getMessage(), 500);
    }
    
    return $results;
});

function validateAccountData(array $acc): array
{
    if (empty($acc['name'])) {
        return ['valid' => false, 'error' => 'Name is required'];
    }
    if (empty($acc['slug'])) {
        return ['valid' => false, 'error' => 'Slug is required'];
    }
    if (!preg_match('/^[a-z0-9-]+$/', $acc['slug'])) {
        return ['valid' => false, 'error' => 'Invalid slug format'];
    }
    
    // التحقق من عدم وجود slug مكرر
    $exists = Db::exists('accounts', 'slug = ?', [$acc['slug']]);
    if ($exists) {
        return ['valid' => false, 'error' => 'Slug already exists'];
    }
    
    return ['valid' => true];
}

// ================================================================
// تشغيل الـ Router
// ================================================================

ApiRouter::run();
