-- ============================================================
-- نظام توزيع أكواد التحقق - قاعدة البيانات الكاملة
-- الإصدار: 1.1
-- التاريخ: 2025-12-01
-- ============================================================
-- 
-- ملاحظات التصميم:
-- ─────────────────
-- 1. جداول الحدود (limits_*):
--    - NULL = لا يوجد حد مخصص، استخدم الطبقة الأدنى
--    - 0 = محظور تماماً (صفر طلبات)
--    - > 0 = الحد الفعلي
--    - الأولوية: user_account > user > superdist > global
--
-- 2. منطق الحظر:
--    - الاعتماد الأساسي على حقل status في users/super_distributors
--    - جدول bans للتاريخ والتفاصيل فقط
--
-- 3. عدادات التوليد:
--    - العدادات في superdist_accounts للتحقق السريع
--    - جدول activation_codes للتقارير والتدقيق
--    - التصفير بإجراءات مخزنة (cron jobs)
--
-- 4. حقل end_at:
--    - يُحسب في PHP عند الإنشاء/التعديل
--    - الصيغة: start_at + duration_days
-- ============================================================

SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
SET time_zone = "+00:00";
SET NAMES utf8mb4;

-- ============================================================
-- 1. جدول الحسابات (Accounts)
-- ============================================================
-- يمثل حسابات الدخول في الموقع، كل حساب له مفتاح سري لتوليد أكواد التحقق

CREATE TABLE `accounts` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `slug` VARCHAR(64) NOT NULL COMMENT 'معرّف نصي فريد (مثل: acc-vip-2025)',
    `name` VARCHAR(128) NOT NULL COMMENT 'الاسم الوصفي للحساب',
    `secret_key` VARCHAR(128) NOT NULL COMMENT 'المفتاح السري لتوليد أكواد التحقق الزمنية',
    `description` TEXT DEFAULT NULL COMMENT 'وصف اختياري للحساب',
    `status` ENUM('active', 'inactive') NOT NULL DEFAULT 'active' COMMENT 'حالة الحساب',
    `default_access_type` ENUM('lifetime', 'timed') NOT NULL DEFAULT 'lifetime' COMMENT 'نوع الاشتراك الافتراضي للزبائن',
    `default_access_days` INT UNSIGNED DEFAULT NULL COMMENT 'عدد أيام الاشتراك الافتراضي (إذا كان timed)',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_slug` (`slug`),
    KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='جدول الحسابات الرئيسية';

-- ============================================================
-- 2. جدول المستخدمين/الزبائن (Users)
-- ============================================================
-- يمثل الزبائن النهائيين الذين يفعّلون الحسابات ويطلبون أكواد التحقق

CREATE TABLE `users` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `telegram_id` BIGINT NOT NULL COMMENT 'معرف تيليجرام للمستخدم',
    `username` VARCHAR(64) DEFAULT NULL COMMENT 'اسم المستخدم في تيليجرام (بدون @)',
    `first_name` VARCHAR(128) DEFAULT NULL COMMENT 'الاسم الأول',
    `last_name` VARCHAR(128) DEFAULT NULL COMMENT 'الاسم الأخير',
    `status` ENUM('active', 'banned') NOT NULL DEFAULT 'active' COMMENT 'حالة المستخدم',
    `ban_reason` VARCHAR(255) DEFAULT NULL COMMENT 'سبب الحظر (إن وجد)',
    `banned_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'تاريخ الحظر',
    `banned_by` VARCHAR(64) DEFAULT NULL COMMENT 'من قام بالحظر (admin/superdist_id)',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_telegram_id` (`telegram_id`),
    KEY `idx_status` (`status`),
    KEY `idx_username` (`username`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='جدول المستخدمين/الزبائن';

-- ============================================================
-- 3. جدول السوبر موزعين (Super Distributors)
-- ============================================================
-- يمثل الموزعين الذين يولّدون أكواد التفعيل ويبيعونها للزبائن

CREATE TABLE `super_distributors` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `telegram_id` BIGINT NOT NULL COMMENT 'معرف تيليجرام للسوبر موزع',
    `username` VARCHAR(64) DEFAULT NULL COMMENT 'اسم المستخدم في تيليجرام',
    `display_name` VARCHAR(128) NOT NULL COMMENT 'الاسم المعروض في التقارير',
    `status` ENUM('active', 'suspended', 'banned', 'archived') NOT NULL DEFAULT 'active' COMMENT 'حالة السوبر موزع',
    `status_reason` VARCHAR(255) DEFAULT NULL COMMENT 'سبب تغيير الحالة',
    `status_changed_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'تاريخ آخر تغيير للحالة',
    
    -- صلاحيات البث
    `can_broadcast` TINYINT(1) NOT NULL DEFAULT 1 COMMENT 'هل يُسمح له بالبث؟',
    `broadcast_content_type` ENUM('text_only', 'text_images', 'all') NOT NULL DEFAULT 'text_only' COMMENT 'أنواع المحتوى المسموحة للبث',
    
    -- صلاحيات إضافية
    `can_ban_users` TINYINT(1) NOT NULL DEFAULT 0 COMMENT 'هل يُسمح له بحظر زبائنه؟',
    
    `notes` TEXT DEFAULT NULL COMMENT 'ملاحظات إدارية',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_telegram_id` (`telegram_id`),
    KEY `idx_status` (`status`),
    KEY `idx_username` (`username`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='جدول السوبر موزعين';

-- ============================================================
-- 4. جدول ربط السوبر موزع بالحسابات (SuperDist_Accounts)
-- ============================================================
-- يربط كل سوبر موزع بالحسابات المسموح له بتوليد أكواد تفعيل لها، مع الفترات والحدود

CREATE TABLE `superdist_accounts` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `superdist_id` INT UNSIGNED NOT NULL COMMENT 'معرف السوبر موزع',
    `account_id` INT UNSIGNED NOT NULL COMMENT 'معرف الحساب',
    
    -- فترة النشاط
    `start_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'بداية فترة النشاط',
    `duration_days` INT UNSIGNED NOT NULL COMMENT 'عدد أيام الصلاحية',
    `end_at` TIMESTAMP NOT NULL COMMENT 'نهاية فترة النشاط (يُحسب في PHP: start_at + duration_days)',
    
    -- حدود توليد أكواد التفعيل (ACT-codes)
    `activation_limit_daily` INT UNSIGNED DEFAULT NULL COMMENT 'حد توليد الأكواد اليومي (NULL = بلا حد)',
    `activation_limit_weekly` INT UNSIGNED DEFAULT NULL COMMENT 'حد توليد الأكواد الأسبوعي',
    `activation_limit_monthly` INT UNSIGNED DEFAULT NULL COMMENT 'حد توليد الأكواد الشهري',
    
    -- عدادات التوليد الفعلية (تُصفَّر دورياً)
    `activations_today` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT 'عدد الأكواد المولّدة اليوم',
    `activations_this_week` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT 'عدد الأكواد المولّدة هذا الأسبوع',
    `activations_this_month` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT 'عدد الأكواد المولّدة هذا الشهر',
    
    -- تتبع آخر تصفير (لضمان idempotency في cron jobs)
    `last_daily_reset` DATE DEFAULT NULL COMMENT 'تاريخ آخر تصفير يومي',
    `last_weekly_reset` CHAR(6) DEFAULT NULL COMMENT 'آخر تصفير أسبوعي (YYYYWW format)',
    `last_monthly_reset` CHAR(6) DEFAULT NULL COMMENT 'آخر تصفير شهري (YYYYMM format)',
    
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_superdist_account` (`superdist_id`, `account_id`),
    KEY `idx_account_id` (`account_id`),
    KEY `idx_end_at` (`end_at`),
    
    CONSTRAINT `fk_sa_superdist` FOREIGN KEY (`superdist_id`) REFERENCES `super_distributors` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    CONSTRAINT `fk_sa_account` FOREIGN KEY (`account_id`) REFERENCES `accounts` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='ربط السوبر موزع بالحسابات مع الفترات والحدود';

-- ============================================================
-- 5. جدول أكواد اشتراك السوبر موزعين (SuperDist_Subscription_Codes)
-- ============================================================
-- أكواد لإنشاء سوبر موزع جديد أو ترقية/تمديد سوبر موزع موجود

CREATE TABLE `superdist_subscription_codes` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `code` VARCHAR(32) NOT NULL COMMENT 'الكود الفريد (مثل: SD-ABCD-1234)',
    `code_type` ENUM('new', 'upgrade') NOT NULL COMMENT 'نوع الكود: new=إنشاء جديد، upgrade=ترقية/تمديد',
    
    -- الحسابات والمدد (JSON array)
    `accounts_data` JSON NOT NULL COMMENT 'بيانات الحسابات [{account_id, duration_days}, ...]',
    
    -- الاستخدامات
    `max_uses` TINYINT UNSIGNED NOT NULL DEFAULT 1 COMMENT 'عدد الاستخدامات المسموحة',
    `current_uses` TINYINT UNSIGNED NOT NULL DEFAULT 0 COMMENT 'عدد الاستخدامات الحالية',
    
    -- الصلاحية
    `expires_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'تاريخ انتهاء صلاحية الكود (NULL = لا ينتهي)',
    
    -- الحالة
    `status` ENUM('unused', 'partial', 'used', 'expired', 'disabled') NOT NULL DEFAULT 'unused' COMMENT 'حالة الكود',
    
    -- من استخدمه
    `used_by_superdist_id` INT UNSIGNED DEFAULT NULL COMMENT 'السوبر موزع الذي استخدم الكود',
    `used_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'تاريخ الاستخدام',
    
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `created_by` VARCHAR(64) DEFAULT 'admin' COMMENT 'من أنشأ الكود',
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_code` (`code`),
    KEY `idx_status` (`status`),
    KEY `idx_code_type` (`code_type`),
    KEY `idx_expires_at` (`expires_at`),
    
    CONSTRAINT `fk_ssc_superdist` FOREIGN KEY (`used_by_superdist_id`) REFERENCES `super_distributors` (`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='أكواد اشتراك السوبر موزعين';

-- ============================================================
-- 6. جدول أكواد التفعيل (Activation_Codes)
-- ============================================================
-- أكواد يولّدها السوبر موزع ليبيعها للزبائن لتفعيل الحسابات

CREATE TABLE `activation_codes` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `code` VARCHAR(32) NOT NULL COMMENT 'الكود الفريد (مثل: ACT-ABCD-1234)',
    `superdist_id` INT UNSIGNED NOT NULL COMMENT 'السوبر موزع الذي ولّد الكود',
    `account_id` INT UNSIGNED NOT NULL COMMENT 'الحساب المرتبط بالكود',
    
    -- نوع الاشتراك عند التفعيل
    `access_type` ENUM('lifetime', 'timed') NOT NULL DEFAULT 'lifetime' COMMENT 'نوع الاشتراك',
    `access_days` INT UNSIGNED DEFAULT NULL COMMENT 'عدد أيام الاشتراك (إذا كان timed)',
    
    -- صلاحية الكود نفسه
    `expires_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'تاريخ انتهاء صلاحية الكود (يمكن أن يكون قبل انتهاء صلاحية السوبر موزع)',
    
    -- الحالة
    `status` ENUM('unused', 'used', 'expired', 'disabled') NOT NULL DEFAULT 'unused' COMMENT 'حالة الكود',
    
    -- بيانات الاستخدام
    `used_by_user_id` INT UNSIGNED DEFAULT NULL COMMENT 'المستخدم الذي فعّل الكود',
    `used_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'تاريخ التفعيل',
    
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_code` (`code`),
    KEY `idx_superdist_id` (`superdist_id`),
    KEY `idx_account_id` (`account_id`),
    KEY `idx_status` (`status`),
    KEY `idx_expires_at` (`expires_at`),
    KEY `idx_superdist_account_status` (`superdist_id`, `account_id`, `status`),
    
    CONSTRAINT `fk_ac_superdist` FOREIGN KEY (`superdist_id`) REFERENCES `super_distributors` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    CONSTRAINT `fk_ac_account` FOREIGN KEY (`account_id`) REFERENCES `accounts` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    CONSTRAINT `fk_ac_user` FOREIGN KEY (`used_by_user_id`) REFERENCES `users` (`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='أكواد التفعيل للزبائن';

-- ============================================================
-- 7. جدول اشتراكات الزبائن في الحسابات (User_Accounts)
-- ============================================================
-- يربط كل مستخدم بالحسابات التي فعّلها

CREATE TABLE `user_accounts` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `user_id` INT UNSIGNED NOT NULL COMMENT 'معرف المستخدم',
    `account_id` INT UNSIGNED NOT NULL COMMENT 'معرف الحساب',
    `superdist_id` INT UNSIGNED DEFAULT NULL COMMENT 'السوبر موزع الذي أعطاه الكود',
    `activation_code_id` INT UNSIGNED DEFAULT NULL COMMENT 'كود التفعيل المستخدم',
    
    -- نوع الاشتراك
    `access_type` ENUM('lifetime', 'timed') NOT NULL DEFAULT 'lifetime' COMMENT 'نوع الاشتراك',
    `access_days` INT UNSIGNED DEFAULT NULL COMMENT 'عدد أيام الاشتراك (إذا كان timed)',
    `activated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ التفعيل',
    `access_expires_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'تاريخ انتهاء الاشتراك (محسوب للـ timed)',
    
    -- حالة الاشتراك
    `status` ENUM('active', 'expired', 'revoked') NOT NULL DEFAULT 'active' COMMENT 'حالة الاشتراك',
    
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_user_account` (`user_id`, `account_id`),
    KEY `idx_account_id` (`account_id`),
    KEY `idx_superdist_id` (`superdist_id`),
    KEY `idx_access_expires_at` (`access_expires_at`),
    KEY `idx_status` (`status`),
    
    CONSTRAINT `fk_ua_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    CONSTRAINT `fk_ua_account` FOREIGN KEY (`account_id`) REFERENCES `accounts` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    CONSTRAINT `fk_ua_superdist` FOREIGN KEY (`superdist_id`) REFERENCES `super_distributors` (`id`) ON DELETE SET NULL ON UPDATE CASCADE,
    CONSTRAINT `fk_ua_activation_code` FOREIGN KEY (`activation_code_id`) REFERENCES `activation_codes` (`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='اشتراكات المستخدمين في الحسابات';

-- ============================================================
-- 8. جدول سجلات طلب أكواد التحقق (Verification_Logs)
-- ============================================================
-- يسجّل كل طلب كود تحقق من المستخدم

CREATE TABLE `verification_logs` (
    `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
    `user_id` INT UNSIGNED NOT NULL COMMENT 'معرف المستخدم',
    `account_id` INT UNSIGNED NOT NULL COMMENT 'معرف الحساب',
    `superdist_id` INT UNSIGNED DEFAULT NULL COMMENT 'السوبر موزع المصدر (للإحصائيات)',
    `verification_code` VARCHAR(16) DEFAULT NULL COMMENT 'الكود المُولَّد (اختياري للتتبع)',
    `requested_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'وقت الطلب',
    `ip_address` VARCHAR(45) DEFAULT NULL COMMENT 'عنوان IP (اختياري)',
    
    PRIMARY KEY (`id`),
    KEY `idx_user_id` (`user_id`),
    KEY `idx_account_id` (`account_id`),
    KEY `idx_superdist_id` (`superdist_id`),
    KEY `idx_requested_at` (`requested_at`),
    KEY `idx_user_account_date` (`user_id`, `account_id`, `requested_at`),
    
    CONSTRAINT `fk_vl_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    CONSTRAINT `fk_vl_account` FOREIGN KEY (`account_id`) REFERENCES `accounts` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    CONSTRAINT `fk_vl_superdist` FOREIGN KEY (`superdist_id`) REFERENCES `super_distributors` (`id`) ON DELETE SET NULL ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='سجلات طلب أكواد التحقق';

-- ============================================================
-- 9. جدول الردود الافتراضية (Replies)
-- ============================================================
-- يحتوي نصوص الرسائل الافتراضية للنظام

CREATE TABLE `replies` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `key` VARCHAR(64) NOT NULL COMMENT 'مفتاح الرد (مثل: WELCOME_USER)',
    `default_text` TEXT NOT NULL COMMENT 'النص الافتراضي',
    `description` VARCHAR(255) DEFAULT NULL COMMENT 'وصف الغرض من هذا الرد',
    `placeholders` VARCHAR(255) DEFAULT NULL COMMENT 'المتغيرات المتاحة (مثل: {name}, {days})',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_key` (`key`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='الردود الافتراضية';

-- ============================================================
-- 10. جدول الردود الخاصة بالسوبر موزعين (SuperDist_Replies)
-- ============================================================
-- ردود مخصصة لزبائن سوبر موزع معين

CREATE TABLE `superdist_replies` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `superdist_id` INT UNSIGNED NOT NULL COMMENT 'معرف السوبر موزع',
    `reply_key` VARCHAR(64) NOT NULL COMMENT 'مفتاح الرد',
    `custom_text` TEXT NOT NULL COMMENT 'النص المخصص',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_superdist_key` (`superdist_id`, `reply_key`),
    
    CONSTRAINT `fk_sr_superdist` FOREIGN KEY (`superdist_id`) REFERENCES `super_distributors` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='ردود مخصصة لسوبر موزع';

-- ============================================================
-- 11. جدول حدود الحساب (Limits_Account) - جديد
-- ============================================================
-- حدود افتراضية لطلب أكواد التحقق على مستوى الحساب
-- الأولوية: 3.5 - بين الحدود العامة وحدود السوبر موزع
-- 
-- يُستخدم لضبط حدود خاصة بحساب معين تنطبق على جميع المستخدمين

CREATE TABLE `limits_account` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `account_id` INT UNSIGNED NOT NULL COMMENT 'معرف الحساب (FK)',
    `daily_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد اليومي (NULL = استخدم العام)',
    `weekly_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد الأسبوعي (NULL = استخدم العام)',
    `monthly_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد الشهري (NULL = استخدم العام)',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_account_id` (`account_id`),
    
    CONSTRAINT `fk_la_account` FOREIGN KEY (`account_id`) REFERENCES `accounts` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='حدود الحساب (أولوية 3.5 - بين global و superdist)';

-- ============================================================
-- 12. جدول الحدود العامة (Limits_Global)
-- ============================================================
-- الحدود الافتراضية لطلب أكواد التحقق (تنطبق على الجميع)
-- الأولوية: 4 (الأدنى) - تُستخدم إذا لم يوجد حد أعلى
-- 
-- قيم الحقول:
--   NULL = غير مسموح (لا تُستخدم هنا لأنه الافتراضي)
--   0 = محظور تماماً
--   > 0 = الحد الفعلي

CREATE TABLE `limits_global` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `limit_type` ENUM('verification') NOT NULL DEFAULT 'verification' COMMENT 'نوع الحد',
    `daily_limit` INT UNSIGNED NOT NULL DEFAULT 2 COMMENT 'الحد اليومي (افتراضي: 2)',
    `weekly_limit` INT UNSIGNED NOT NULL DEFAULT 10 COMMENT 'الحد الأسبوعي (افتراضي: 10)',
    `monthly_limit` INT UNSIGNED NOT NULL DEFAULT 30 COMMENT 'الحد الشهري (افتراضي: 30)',
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_limit_type` (`limit_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='الحدود العامة للنظام (أولوية 4 - الأدنى)';

-- ============================================================
-- 12. جدول حدود السوبر موزع (Limits_SuperDist)
-- ============================================================
-- حدود خاصة تنطبق على زبائن سوبر موزع معين
-- الأولوية: 3 - تُستخدم إذا لم يوجد حد للمستخدم
--
-- قيم الحقول:
--   NULL = لا يوجد حد مخصص، استخدم الطبقة الأدنى (global)
--   0 = محظور تماماً
--   > 0 = الحد الفعلي

CREATE TABLE `limits_superdist` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `superdist_id` INT UNSIGNED NOT NULL COMMENT 'معرف السوبر موزع (FK)',
    `daily_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد اليومي (NULL = استخدم العام)',
    `weekly_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد الأسبوعي (NULL = استخدم العام)',
    `monthly_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد الشهري (NULL = استخدم العام)',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_superdist_id` (`superdist_id`),
    
    CONSTRAINT `fk_ls_superdist` FOREIGN KEY (`superdist_id`) REFERENCES `super_distributors` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='حدود زبائن سوبر موزع (أولوية 3)';

-- ============================================================
-- 13. جدول حدود المستخدم (Limits_User)
-- ============================================================
-- حدود خاصة بمستخدم معين (على مستوى كل حساباته مجتمعة)
-- الأولوية: 2 - تُستخدم إذا لم يوجد حد للمستخدم+حساب
--
-- قيم الحقول:
--   NULL = لا يوجد حد مخصص، استخدم الطبقة الأدنى (superdist أو global)
--   0 = محظور تماماً
--   > 0 = الحد الفعلي

CREATE TABLE `limits_user` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `user_id` INT UNSIGNED NOT NULL COMMENT 'معرف المستخدم (FK)',
    `daily_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد اليومي (NULL = استخدم الأدنى)',
    `weekly_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد الأسبوعي (NULL = استخدم الأدنى)',
    `monthly_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد الشهري (NULL = استخدم الأدنى)',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_user_id` (`user_id`),
    
    CONSTRAINT `fk_lu_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='حدود المستخدم (أولوية 2)';

-- ============================================================
-- 14. جدول حدود المستخدم + الحساب (Limits_User_Account)
-- ============================================================
-- حدود خاصة لمستخدم معين على حساب معين
-- الأولوية: 1 (الأعلى) - تُستخدم أولاً إذا وُجدت
--
-- قيم الحقول:
--   NULL = لا يوجد حد مخصص، استخدم الطبقة الأدنى
--   0 = محظور تماماً من هذا الحساب
--   > 0 = الحد الفعلي

CREATE TABLE `limits_user_account` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `user_id` INT UNSIGNED NOT NULL COMMENT 'معرف المستخدم (FK)',
    `account_id` INT UNSIGNED NOT NULL COMMENT 'معرف الحساب (FK)',
    `daily_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد اليومي (NULL = استخدم الأدنى)',
    `weekly_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد الأسبوعي (NULL = استخدم الأدنى)',
    `monthly_limit` INT UNSIGNED DEFAULT NULL COMMENT 'الحد الشهري (NULL = استخدم الأدنى)',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_user_account` (`user_id`, `account_id`),
    KEY `idx_account_id` (`account_id`),
    
    CONSTRAINT `fk_lua_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
    CONSTRAINT `fk_lua_account` FOREIGN KEY (`account_id`) REFERENCES `accounts` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='حدود المستخدم+الحساب (أولوية 1 - الأعلى)';

-- ============================================================
-- 15. جدول الحظر (Bans)
-- ============================================================
-- سجل الحظر للمستخدمين والسوبر موزعين
--
-- ملاحظة مهمة - منطق الحظر:
-- ─────────────────────────────
-- • الاعتماد الأساسي: على حقل status في جدول users/super_distributors
-- • هذا الجدول: للتاريخ والتفاصيل والتدقيق فقط
--
-- عند الحظر:
--   1. UPDATE users/super_distributors SET status = 'banned'
--   2. INSERT INTO bans (is_active = 1)
--
-- عند فك الحظر:
--   1. UPDATE users/super_distributors SET status = 'active'
--   2. UPDATE bans SET is_active = 0, unbanned_at = NOW()
--
-- دالة isUserBanned(): تتحقق من users.status = 'banned' فقط

CREATE TABLE `bans` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `target_type` ENUM('user', 'superdist') NOT NULL COMMENT 'نوع المحظور',
    `target_id` INT UNSIGNED NOT NULL COMMENT 'معرف المحظور في جدوله',
    `target_telegram_id` BIGINT NOT NULL COMMENT 'معرف تيليجرام للمحظور',
    `reason` VARCHAR(255) DEFAULT NULL COMMENT 'سبب الحظر',
    `banned_by` VARCHAR(64) NOT NULL DEFAULT 'admin' COMMENT 'من قام بالحظر (admin أو superdist_123)',
    `banned_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الحظر',
    `unbanned_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'تاريخ فك الحظر (إن وجد)',
    `unbanned_by` VARCHAR(64) DEFAULT NULL COMMENT 'من فك الحظر',
    `is_active` TINYINT(1) NOT NULL DEFAULT 1 COMMENT 'هل الحظر فعّال (للسجل التاريخي)',
    
    PRIMARY KEY (`id`),
    KEY `idx_target` (`target_type`, `target_id`),
    KEY `idx_telegram_id` (`target_telegram_id`),
    KEY `idx_is_active` (`is_active`),
    KEY `idx_banned_at` (`banned_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='سجل الحظر (للتاريخ والتفاصيل)';

-- ============================================================
-- 16. جدول سجلات البث (Broadcast_Logs)
-- ============================================================
-- يسجّل كل عملية بث جماعي

CREATE TABLE `broadcast_logs` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `broadcaster_type` ENUM('admin', 'superdist') NOT NULL COMMENT 'من قام بالبث',
    `broadcaster_id` VARCHAR(64) NOT NULL COMMENT 'معرف القائم بالبث',
    `audience_type` ENUM('all', 'users', 'superdists', 'superdist_users', 'account_users') NOT NULL COMMENT 'نوع الجمهور',
    `audience_filter` VARCHAR(128) DEFAULT NULL COMMENT 'فلتر إضافي (مثل superdist_id أو account_slug)',
    `content_type` ENUM('text', 'photo', 'video', 'document', 'audio', 'voice', 'sticker', 'animation') NOT NULL COMMENT 'نوع المحتوى',
    `content_text` TEXT DEFAULT NULL COMMENT 'نص الرسالة (إن وجد)',
    `content_file_id` VARCHAR(255) DEFAULT NULL COMMENT 'معرف الملف في تيليجرام',
    `total_recipients` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT 'إجمالي المستلمين المستهدفين',
    `successful_sends` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT 'عدد الإرسالات الناجحة',
    `failed_sends` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT 'عدد الإرسالات الفاشلة',
    `status` ENUM('pending', 'in_progress', 'completed', 'cancelled') NOT NULL DEFAULT 'pending' COMMENT 'حالة البث',
    `started_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'وقت بدء البث',
    `completed_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'وقت انتهاء البث',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    KEY `idx_broadcaster` (`broadcaster_type`, `broadcaster_id`),
    KEY `idx_audience_type` (`audience_type`),
    KEY `idx_status` (`status`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='سجلات البث الجماعي';

-- ============================================================
-- 17. جدول حالات المستخدم المؤقتة (User_States)
-- ============================================================
-- لتخزين حالات الجلسات (مثل: في انتظار محتوى بث)

CREATE TABLE `user_states` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `telegram_id` BIGINT NOT NULL COMMENT 'معرف تيليجرام',
    `user_type` ENUM('admin', 'superdist', 'user') NOT NULL COMMENT 'نوع المستخدم',
    `state` VARCHAR(64) NOT NULL COMMENT 'الحالة الحالية (مثل: awaiting_broadcast_content)',
    `state_data` JSON DEFAULT NULL COMMENT 'بيانات الحالة (مثل: نوع الجمهور، المحتوى المؤقت)',
    `expires_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'انتهاء صلاحية الحالة',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_telegram_id` (`telegram_id`),
    KEY `idx_state` (`state`),
    KEY `idx_expires_at` (`expires_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='حالات الجلسات المؤقتة';

-- ============================================================
-- 18. جدول سجل الأحداث (Event_Logs)
-- ============================================================
-- سجل عام للأحداث المهمة للتتبع والتدقيق
--
-- أنواع الأحداث (event_type):
-- ─────────────────────────────
-- المستخدمون:
--   • user_registered      - تسجيل مستخدم جديد
--   • user_banned          - حظر مستخدم
--   • user_unbanned        - فك حظر مستخدم
--
-- السوبر موزعون:
--   • superdist_created    - إنشاء سوبر موزع
--   • superdist_activated  - تفعيل سوبر موزع
--   • superdist_suspended  - إيقاف سوبر موزع
--   • superdist_banned     - حظر سوبر موزع
--   • superdist_unbanned   - فك حظر سوبر موزع
--   • superdist_archived   - أرشفة سوبر موزع
--
-- الأكواد:
--   • code_generated       - توليد كود تفعيل
--   • code_activated       - تفعيل كود (زبون)
--   • code_expired         - انتهاء كود
--   • verification_requested - طلب كود تحقق
--
-- البث:
--   • broadcast_started    - بدء بث
--   • broadcast_completed  - اكتمال بث
--   • broadcast_cancelled  - إلغاء بث
--
-- الحدود:
--   • limit_exceeded       - تجاوز حد
--   • limit_updated        - تحديث حد
--
-- النظام:
--   • error                - خطأ في النظام
--   • cron_executed        - تنفيذ مهمة مجدولة

CREATE TABLE `event_logs` (
    `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
    `event_type` VARCHAR(64) NOT NULL COMMENT 'نوع الحدث (انظر التوثيق أعلاه)',
    `actor_type` ENUM('admin', 'superdist', 'user', 'system') NOT NULL COMMENT 'من قام بالحدث',
    `actor_id` VARCHAR(64) DEFAULT NULL COMMENT 'معرف الفاعل (telegram_id أو superdist_id)',
    `target_type` VARCHAR(32) DEFAULT NULL COMMENT 'نوع الهدف (user, superdist, account, code)',
    `target_id` VARCHAR(64) DEFAULT NULL COMMENT 'معرف الهدف',
    `details` JSON DEFAULT NULL COMMENT 'تفاصيل إضافية بصيغة JSON',
    `ip_address` VARCHAR(45) DEFAULT NULL COMMENT 'عنوان IP (اختياري)',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    
    PRIMARY KEY (`id`),
    KEY `idx_event_type` (`event_type`),
    KEY `idx_actor` (`actor_type`, `actor_id`),
    KEY `idx_target` (`target_type`, `target_id`),
    KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='سجل الأحداث العام للتتبع والتدقيق';

-- ============================================================
-- إدخال البيانات الافتراضية
-- ============================================================

-- الحدود العامة الافتراضية
INSERT INTO `limits_global` (`limit_type`, `daily_limit`, `weekly_limit`, `monthly_limit`) VALUES
('verification', 2, 10, 30);

-- الردود الافتراضية
INSERT INTO `replies` (`key`, `default_text`, `description`, `placeholders`) VALUES
('WELCOME_USER', 'مرحباً بك! 👋\n\nأرسل كود التفعيل الذي استلمته لتفعيل حسابك.\n\nأو أرسل /help لعرض الأوامر المتاحة.', 'رسالة الترحيب للمستخدم الجديد', NULL),
('WELCOME_SUPERDIST', 'مرحباً بك أيها الموزع! 👋\n\nأرسل /help لعرض أوامرك المتاحة.\nأرسل /my_accounts لعرض حساباتك.', 'رسالة الترحيب للسوبر موزع', NULL),
('ERROR_NOT_IN_CHANNEL', '⚠️ يجب أن تكون مشتركاً في قناتنا أولاً.\n\nاشترك في القناة ثم حاول مجدداً.', 'خطأ: غير مشترك بالقناة', NULL),
('ERROR_INVALID_CODE', '❌ الكود غير صحيح أو غير موجود.\n\nتأكد من كتابة الكود بشكل صحيح.', 'خطأ: كود غير صالح', NULL),
('ERROR_CODE_USED', '❌ هذا الكود مُستخدم مسبقاً.', 'خطأ: الكود مستخدم', NULL),
('ERROR_CODE_EXPIRED', '❌ انتهت صلاحية هذا الكود.', 'خطأ: الكود منتهي', NULL),
('ERROR_CODE_DISABLED', '❌ هذا الكود معطّل.', 'خطأ: الكود معطل', NULL),
('ERROR_NO_ACTIVE_ACCOUNTS_SUPERDIST', '⚠️ لا تملك حسابات نشطة حالياً.\n\nتواصل مع الإدارة لتجديد اشتراكك.', 'خطأ: سوبر موزع بدون حسابات نشطة', NULL),
('ERROR_ACCOUNT_NOT_ACTIVE', '❌ هذا الحساب غير نشط لديك أو انتهت صلاحيتك عليه.', 'خطأ: الحساب غير نشط للسوبر موزع', NULL),
('ERROR_ACTIVATION_LIMIT_DAILY', '⚠️ وصلت للحد اليومي لتوليد الأكواد لهذا الحساب.\n\nحاول غداً.', 'خطأ: تجاوز حد توليد الأكواد اليومي', NULL),
('ERROR_ACTIVATION_LIMIT_WEEKLY', '⚠️ وصلت للحد الأسبوعي لتوليد الأكواد لهذا الحساب.', 'خطأ: تجاوز حد توليد الأكواد الأسبوعي', NULL),
('ERROR_ACTIVATION_LIMIT_MONTHLY', '⚠️ وصلت للحد الشهري لتوليد الأكواد لهذا الحساب.', 'خطأ: تجاوز حد توليد الأكواد الشهري', NULL),
('ERROR_LIMIT_DAILY', '⚠️ وصلت للحد اليومي لطلب أكواد التحقق.\n\nحاول غداً.', 'خطأ: تجاوز حد طلب أكواد التحقق اليومي', '{limit}'),
('ERROR_LIMIT_WEEKLY', '⚠️ وصلت للحد الأسبوعي لطلب أكواد التحقق.', 'خطأ: تجاوز الحد الأسبوعي', '{limit}'),
('ERROR_LIMIT_MONTHLY', '⚠️ وصلت للحد الشهري لطلب أكواد التحقق.', 'خطأ: تجاوز الحد الشهري', '{limit}'),
('ERROR_ACCOUNT_EXPIRED', '⚠️ انتهت مدة اشتراكك في هذا الحساب.\n\nتحتاج إلى شراء مفتاح جديد لتجديد الوصول.', 'خطأ: انتهاء اشتراك المستخدم بالحساب', NULL),
('ERROR_ACCOUNT_NOT_OWNED', '❌ أنت لا تملك هذا الحساب.\n\nاستخدم /myaccounts لعرض حساباتك.', 'خطأ: المستخدم لا يملك الحساب', NULL),
('ERROR_USER_BANNED', '🚫 أنت محظور من استخدام هذا البوت.', 'خطأ: المستخدم محظور', NULL),
('ERROR_SUPERDIST_SUSPENDED', '⚠️ حسابك كموزع موقوف مؤقتاً.\n\nتواصل مع الإدارة.', 'خطأ: سوبر موزع موقوف', NULL),
('ERROR_SUPERDIST_BANNED', '🚫 حسابك كموزع محظور.\n\nتواصل مع الإدارة.', 'خطأ: سوبر موزع محظور', NULL),
('ERROR_SUPERDIST_ARCHIVED', '📦 حسابك كموزع مؤرشف.\n\nتواصل مع الإدارة لإعادة التفعيل.', 'خطأ: سوبر موزع مؤرشف', NULL),
('SUCCESS_ACTIVATION', '✅ تم تفعيل الحساب بنجاح!\n\nالحساب: {account_name}\nنوع الاشتراك: {access_type}\n{expiry_info}\n\nاستخدم: get_code {account_slug}\nلطلب كود التحقق.', 'نجاح تفعيل الحساب', '{account_name}, {account_slug}, {access_type}, {expiry_info}'),
('SUCCESS_VERIFICATION_CODE', '🔐 كود التحقق لحساب {account_name}:\n\n<code>{code}</code>\n\n⏱ صالح لمدة 30 ثانية.', 'كود التحقق', '{account_name}, {code}'),
('SUCCESS_CODE_GENERATED', '✅ تم توليد كود تفعيل:\n\n<code>{code}</code>\n\nللحساب: {account_name}', 'نجاح توليد كود تفعيل', '{code}, {account_name}'),
('SUCCESS_CODES_GENERATED', '✅ تم توليد {count} كود للحساب: {account_name}\n\nالأكواد مرفقة في الملف.', 'نجاح توليد مجموعة أكواد', '{count}, {account_name}'),
('INFO_BROADCAST_AWAITING_CONTENT', '📢 أرسل الآن المحتوى الذي تريد بثّه.\n\nأو أرسل /cancel للإلغاء.', 'انتظار محتوى البث', NULL),
('INFO_BROADCAST_CONFIRM', '📢 تم استلام المحتوى.\n\nنوع المحتوى: {content_type}\nعدد المستلمين: {recipients_count}\n\nأرسل "تأكيد" للإرسال.\nأرسل "تعديل" لتغيير المحتوى.\nأرسل /cancel للإلغاء.', 'تأكيد البث', '{content_type}, {recipients_count}'),
('INFO_BROADCAST_STARTED', '🚀 بدأ البث...\n\nسيتم إرسال الرسائل على دفعات.', 'بدء البث', NULL),
('INFO_BROADCAST_DONE', '✅ اكتمل البث!\n\nنجح: {success}\nفشل: {failed}\nالإجمالي: {total}', 'انتهاء البث', '{success}, {failed}, {total}'),
('INFO_BROADCAST_CANCELLED', '❌ تم إلغاء البث.', 'إلغاء البث', NULL),
('HELP_USER', '📋 <b>الأوامر المتاحة:</b>\n\n<b>تفعيل الحسابات:</b>\n• أرسل كود التفعيل (ACT-...) مباشرة\n\n<b>الحسابات:</b>\n• <code>/myaccounts</code> - عرض حساباتك\n• <code>/account_info [slug]</code> - تفاصيل حساب\n\n<b>طلب كود التحقق:</b>\n• <code>get_code [slug]</code> - أو أرسل الـ slug مباشرة', 'مساعدة المستخدم', NULL),
('HELP_SUPERDIST', '📋 <b>أوامر الموزع:</b>\n\n<b>عرض المعلومات:</b>\n• <code>/my_accounts</code> - حساباتك المرتبطة\n• <code>/mystatus</code> - حالتك وإحصائياتك\n• <code>/my_remaining</code> - الحدود المتبقية\n\n<b>توليد الأكواد:</b>\n• <code>gen_code [slug]</code> - كود واحد\n• <code>gen_codes [slug] [count]</code> - مجموعة أكواد\n• <code>/my_codes [slug] [page]</code> - عرض أكوادك\n• <code>/check_code [code]</code> - فحص كود\n\n<b>البث:</b>\n• <code>/broadcast</code> - بث لجميع زبائنك\n• <code>/broadcast_my_account [slug]</code> - بث لزبائن حساب معين\n\n<b>كود الاشتراك:</b>\n• أرسل الكود (SD-...) لتفعيل/تجديد حساباتك', 'مساعدة السوبر موزع', NULL);

-- ============================================================
-- إجراءات مخزنة (Stored Procedures)
-- ============================================================

DELIMITER $$

-- إجراء لتصفير عدادات توليد الأكواد يومياً
CREATE PROCEDURE `reset_daily_activation_counters`()
BEGIN
    UPDATE `superdist_accounts`
    SET `activations_today` = 0,
        `last_activation_reset` = CURDATE()
    WHERE `last_activation_reset` IS NULL OR `last_activation_reset` < CURDATE();
END$$

-- إجراء لتصفير عدادات توليد الأكواد أسبوعياً (يُنفَّذ كل اثنين)
CREATE PROCEDURE `reset_weekly_activation_counters`()
BEGIN
    UPDATE `superdist_accounts`
    SET `activations_this_week` = 0;
END$$

-- إجراء لتصفير عدادات توليد الأكواد شهرياً (يُنفَّذ أول كل شهر)
CREATE PROCEDURE `reset_monthly_activation_counters`()
BEGIN
    UPDATE `superdist_accounts`
    SET `activations_this_month` = 0;
END$$

-- إجراء لتحويل الأكواد المنتهية إلى حالة expired
CREATE PROCEDURE `expire_outdated_codes`()
BEGIN
    -- أكواد التفعيل التي انتهت صلاحيتها مباشرة
    UPDATE `activation_codes`
    SET `status` = 'expired'
    WHERE `status` = 'unused'
      AND `expires_at` IS NOT NULL
      AND `expires_at` < NOW();
    
    -- أكواد التفعيل لسوبر موزعين انتهت صلاحية حساباتهم
    UPDATE `activation_codes` ac
    INNER JOIN `superdist_accounts` sa ON ac.superdist_id = sa.superdist_id AND ac.account_id = sa.account_id
    SET ac.status = 'expired'
    WHERE ac.status = 'unused'
      AND sa.end_at < NOW();
    
    -- أكواد اشتراك السوبر موزعين المنتهية
    UPDATE `superdist_subscription_codes`
    SET `status` = 'expired'
    WHERE `status` IN ('unused', 'partial')
      AND `expires_at` IS NOT NULL
      AND `expires_at` < NOW();
END$$

-- إجراء لتعطيل أكواد سوبر موزع موقوف/محظور
CREATE PROCEDURE `disable_superdist_codes`(IN p_superdist_id INT)
BEGIN
    UPDATE `activation_codes`
    SET `status` = 'disabled'
    WHERE `superdist_id` = p_superdist_id
      AND `status` = 'unused';
END$$

-- إجراء لحساب إحصائيات النظام
CREATE PROCEDURE `get_system_stats`()
BEGIN
    SELECT
        (SELECT COUNT(*) FROM `accounts` WHERE `status` = 'active') AS active_accounts,
        (SELECT COUNT(*) FROM `super_distributors` WHERE `status` = 'active') AS active_superdists,
        (SELECT COUNT(*) FROM `super_distributors` WHERE `status` = 'suspended') AS suspended_superdists,
        (SELECT COUNT(*) FROM `super_distributors` WHERE `status` = 'banned') AS banned_superdists,
        (SELECT COUNT(*) FROM `users` WHERE `status` = 'active') AS active_users,
        (SELECT COUNT(*) FROM `users` WHERE `status` = 'banned') AS banned_users,
        (SELECT COUNT(*) FROM `superdist_subscription_codes` WHERE `status` = 'unused') AS unused_sd_codes,
        (SELECT COUNT(*) FROM `superdist_subscription_codes` WHERE `status` = 'used') AS used_sd_codes,
        (SELECT COUNT(*) FROM `activation_codes` WHERE `status` = 'unused') AS unused_act_codes,
        (SELECT COUNT(*) FROM `activation_codes` WHERE `status` = 'used') AS used_act_codes,
        (SELECT COUNT(*) FROM `activation_codes` WHERE `status` = 'expired') AS expired_act_codes,
        (SELECT COUNT(*) FROM `verification_logs`) AS total_verifications,
        (SELECT COUNT(*) FROM `verification_logs` WHERE DATE(`requested_at`) = CURDATE()) AS today_verifications,
        (SELECT COUNT(*) FROM `verification_logs` WHERE `requested_at` >= DATE_SUB(NOW(), INTERVAL 7 DAY)) AS week_verifications,
        (SELECT COUNT(*) FROM `verification_logs` WHERE `requested_at` >= DATE_SUB(NOW(), INTERVAL 30 DAY)) AS month_verifications;
END$$

DELIMITER ;

-- ============================================================
-- فهارس إضافية للأداء
-- ============================================================

-- فهرس مركب لتسريع البحث عن أكواد التحقق حسب الفترة
CREATE INDEX `idx_vl_user_account_period` ON `verification_logs` (`user_id`, `account_id`, `requested_at`);

-- فهرس لتسريع البحث عن أكواد التفعيل حسب السوبر موزع والحالة
CREATE INDEX `idx_ac_superdist_status_created` ON `activation_codes` (`superdist_id`, `status`, `created_at`);

-- ============================================================
-- نهاية الملف
-- ============================================================
