<?php
/**
 * ============================================================
 * ملف التحميل (Bootstrap)
 * ============================================================
 * 
 * يحمّل جميع المكتبات والإعدادات ويهيّئها.
 * يجب استدعاء هذا الملف في بداية كل نقطة دخول (webhook).
 * 
 * الاستخدام:
 *   require __DIR__ . '/libs/bootstrap.php';
 */

// منع الوصول المباشر
if (!defined('BOT_STARTED')) {
    http_response_code(403);
    exit('Direct access not allowed');
}

// ============================================================
// تحميل الإعدادات
// ============================================================

$config = require __DIR__ . '/../config/env.php';
$limitsConfig = require __DIR__ . '/../config/limits.php';

// ضبط المنطقة الزمنية
date_default_timezone_set($config['TIMEZONE'] ?? 'Asia/Baghdad');

// ضبط معالجة الأخطاء
if ($config['DEBUG_MODE'] ?? false) {
    error_reporting(E_ALL);
    ini_set('display_errors', '1');
} else {
    error_reporting(E_ERROR | E_PARSE);
    ini_set('display_errors', '0');
}

// ============================================================
// تحميل المكتبات
// ============================================================

require_once __DIR__ . '/Db.php';
require_once __DIR__ . '/Telegram.php';
require_once __DIR__ . '/CodeGenerator.php';
require_once __DIR__ . '/Limits.php';
require_once __DIR__ . '/Replies.php';
require_once __DIR__ . '/Logger.php';
require_once __DIR__ . '/State.php';

// المكتبات الجديدة (V2)
require_once __DIR__ . '/Voucher.php';
require_once __DIR__ . '/SystemChecker.php';
require_once __DIR__ . '/BroadcastFilter.php';
require_once __DIR__ . '/Settings.php';
require_once __DIR__ . '/ProcessCodesWindow.php';
require_once __DIR__ . '/BatchAccessManager.php';

// ============================================================
// تهيئة المكتبات
// ============================================================

Db::init($config);
Telegram::init($config, defined('BOT_TYPE') ? BOT_TYPE : 'public');
CodeGenerator::init($config);
Limits::init($limitsConfig);
Logger::init($config);

// ============================================================
// دوال مساعدة عامة
// ============================================================

/**
 * الحصول على الإعدادات
 * 
 * @param string|null $key المفتاح (اختياري)
 * @param mixed $default القيمة الافتراضية
 * @return mixed
 */
function config(?string $key = null, $default = null)
{
    global $config;
    
    if ($key === null) {
        return $config;
    }
    
    return $config[$key] ?? $default;
}

/**
 * التحقق من أن المرسل أدمن
 * 
 * @param int $telegramId
 * @return bool
 */
function isAdmin(int $telegramId): bool
{
    $adminIds = config('ADMIN_IDS', []);
    return in_array($telegramId, $adminIds, true);
}

/**
 * التحقق من أن المرسل سوبر موزع
 * 
 * @param int $telegramId
 * @return array|null بيانات السوبر موزع إذا وُجد
 */
function getSuperDistByTelegramId(int $telegramId): ?array
{
    return Db::fetchOne(
        "SELECT * FROM super_distributors WHERE telegram_id = ?",
        [$telegramId]
    );
}

/**
 * التحقق من أن المرسل مستخدم عادي
 * 
 * @param int $telegramId
 * @return array|null بيانات المستخدم إذا وُجد
 */
function getUserByTelegramId(int $telegramId): ?array
{
    return Db::fetchOne(
        "SELECT * FROM users WHERE telegram_id = ?",
        [$telegramId]
    );
}

/**
 * إنشاء أو جلب مستخدم
 * 
 * @param int $telegramId
 * @param string|null $username
 * @param string|null $firstName
 * @param string|null $lastName
 * @return array
 */
function getOrCreateUser(int $telegramId, ?string $username = null, ?string $firstName = null, ?string $lastName = null): array
{
    $user = getUserByTelegramId($telegramId);
    
    if ($user) {
        // تحديث المعلومات إذا تغيرت
        if ($username !== $user['username'] || $firstName !== $user['first_name']) {
            Db::update('users', [
                'username'   => $username,
                'first_name' => $firstName,
                'last_name'  => $lastName,
            ], 'id = ?', [$user['id']]);
        }
        return $user;
    }
    
    // إنشاء مستخدم جديد
    $id = Db::insert('users', [
        'telegram_id' => $telegramId,
        'username'    => $username,
        'first_name'  => $firstName,
        'last_name'   => $lastName,
        'status'      => 'active',
    ]);
    
    Logger::event('user_registered', 'system', null, 'user', $id, [
        'telegram_id' => $telegramId,
        'username'    => $username,
    ]);
    
    return Db::fetchOne("SELECT * FROM users WHERE id = ?", [$id]);
}

/**
 * التحقق من حظر المستخدم
 * 
 * @param int|string $userId معرف المستخدم في جدول users
 * @return bool
 */
function isUserBanned(int|string $userId): bool
{
    $userId = (int) $userId;
    if ($userId <= 0) {
        return false;
    }
    $status = Db::fetchValue("SELECT status FROM users WHERE id = ?", [$userId]);
    return $status === 'banned';
}

/**
 * التحقق من حظر السوبر موزع
 * 
 * @param int|string $superdistId
 * @return bool
 */
function isSuperDistBanned(int|string $superdistId): bool
{
    $superdistId = (int) $superdistId;
    if ($superdistId <= 0) {
        return false;
    }
    $status = Db::fetchValue("SELECT status FROM super_distributors WHERE id = ?", [$superdistId]);
    return in_array($status, ['banned', 'suspended'], true);
}

/**
 * التحقق من الاشتراك بالقناة
 * 
 * @param int $telegramId
 * @return bool
 */
function checkChannelSubscription(int $telegramId): bool
{
    return Telegram::isSubscribedToChannel($telegramId);
}

/**
 * إرسال رسالة خطأ الاشتراك بالقناة
 * 
 * @param int $chatId
 */
function sendChannelSubscriptionError(int $chatId): void
{
    $channelLink = config('CHANNEL_LINK', '#');
    $message = Replies::get('ERROR_NOT_IN_CHANNEL');
    $message .= "\n\n🔗 " . $channelLink;
    
    Telegram::sendMessage($chatId, $message);
}

/**
 * جلب حساب بالـ slug
 * 
 * @param string $slug
 * @return array|null
 */
function getAccountBySlug(string $slug): ?array
{
    return Db::fetchOne("SELECT * FROM accounts WHERE slug = ?", [$slug]);
}

/**
 * جلب حساب بالمعرف
 * 
 * @param int $id
 * @return array|null
 */
function getAccountById(int $id): ?array
{
    return Db::fetchOne("SELECT * FROM accounts WHERE id = ?", [$id]);
}

/**
 * تنسيق التاريخ للعرض
 * 
 * @param string $date
 * @return string
 */
function formatDate(string $date): string
{
    return date('Y/m/d H:i', strtotime($date));
}

/**
 * تنسيق عدد الأيام المتبقية
 * 
 * @param string $endDate
 * @return string
 */
function formatRemainingDays(string $endDate): string
{
    $remaining = (strtotime($endDate) - time()) / 86400;
    
    if ($remaining < 0) {
        return 'منتهي';
    }
    
    if ($remaining < 1) {
        $hours = ceil($remaining * 24);
        return "{$hours} ساعة";
    }
    
    return ceil($remaining) . ' يوم';
}
