<?php
/**
 * ============================================================
 * مكتبة Telegram API
 * ============================================================
 * 
 * توفر دوال للتعامل مع Telegram Bot API.
 * تدعم إرسال الرسائل النصية والوسائط المتعددة.
 * 
 * الاستخدام:
 *   Telegram::init($config);
 *   Telegram::sendMessage($chatId, "مرحباً!");
 */

class Telegram
{
    /** @var string توكن البوت الحالي */
    private static string $token = '';
    
    /** @var array إعدادات النظام */
    private static array $config = [];
    
    /** @var int مهلة الاتصال بالثواني */
    private static int $timeout = 30;
    
    /**
     * تهيئة المكتبة
     * 
     * @param array $config إعدادات من env.php
     * @param string $botType نوع البوت: 'admin' أو 'public'
     */
    public static function init(array $config, string $botType = 'public'): void
    {
        self::$config = $config;
        self::$token = ($botType === 'admin') 
            ? $config['ADMIN_BOT_TOKEN'] 
            : $config['PUBLIC_BOT_TOKEN'];
    }
    
    /**
     * تعيين التوكن مباشرة
     * 
     * @param string $token
     */
    public static function setToken(string $token): void
    {
        self::$token = $token;
    }
    
    /**
     * استدعاء Telegram API
     * 
     * ============================================================
     * التعامل مع أخطاء Flood Control (429):
     * ============================================================
     * - عند استلام خطأ 429، يتم استخراج retry_after من الاستجابة
     * - الانتظار للمدة المحددة ثم إعادة المحاولة
     * - الحد الأقصى لإعادة المحاولات: 3
     * 
     * @param string $method اسم الدالة
     * @param array $params المعاملات
     * @param int $retryCount عدد المحاولات الحالية
     * @return array الاستجابة
     */
    public static function apiCall(string $method, array $params = [], int $retryCount = 0): array
    {
        $url = "https://api.telegram.org/bot" . self::$token . "/{$method}";
        
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => $params,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_TIMEOUT        => self::$timeout,
        ]);
        
        $response = curl_exec($ch);
        
        if ($response === false) {
            $error = curl_error($ch);
            curl_close($ch);
            return ['ok' => false, 'error' => $error, 'error_code' => 0];
        }
        
        curl_close($ch);
        
        $result = json_decode($response, true);
        if (!is_array($result)) {
            return ['ok' => false, 'error' => 'Invalid JSON response', 'raw' => $response];
        }
        
        // ============================================================
        // التعامل مع خطأ 429 (Flood Control / Too Many Requests)
        // ============================================================
        if (($result['ok'] ?? true) === false && ($result['error_code'] ?? 0) === 429) {
            $retryAfter = $result['parameters']['retry_after'] ?? 5;
            
            // الحد الأقصى 3 محاولات
            if ($retryCount < 3) {
                // تسجيل الحدث (اختياري)
                if (class_exists('Logger')) {
                    Logger::error("Telegram Flood Control: waiting {$retryAfter}s (attempt " . ($retryCount + 1) . "/3)", [
                        'method' => $method,
                        'retry_after' => $retryAfter,
                    ]);
                }
                
                // الانتظار ثم إعادة المحاولة
                sleep($retryAfter + 1); // +1 للأمان
                return self::apiCall($method, $params, $retryCount + 1);
            }
        }
        
        return $result;
    }
    
    /**
     * إرسال رسالة نصية
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $text النص
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function sendMessage($chatId, string $text, array $extra = []): array
    {
        $params = array_merge([
            'chat_id'                  => $chatId,
            'text'                     => $text,
            'parse_mode'               => 'HTML',
            'disable_web_page_preview' => true,
        ], $extra);
        
        return self::apiCall('sendMessage', $params);
    }
    
    /**
     * إرسال رسالة طويلة (تقسيم تلقائي)
     * 
     * ============================================================
     * المواصفات:
     * ============================================================
     * - الحد الأقصى لتيليجرام: 4096 حرف
     * - الحد الآمن المستخدم: 3800 حرف (هامش أمان للـ HTML tags والترقيم)
     * - يتم إرسال الأجزاء بالترتيب الصحيح مع ترقيم (1/N, 2/N, ...)
     * - تأخير 100ms بين كل جزء لتجنب rate limiting
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $text النص
     * @param array $extra معاملات إضافية
     * @param int $maxLength الحد الأقصى لطول الرسالة (افتراضي: 3800 - آمن)
     * @return array نتائج الإرسال لكل جزء
     */
    public static function sendLongMessage($chatId, string $text, array $extra = [], int $maxLength = 3800): array
    {
        if (mb_strlen($text) <= $maxLength) {
            return [self::sendMessage($chatId, $text, $extra)];
        }
        
        $parts = self::splitMessage($text, $maxLength);
        $results = [];
        
        foreach ($parts as $i => $part) {
            if (count($parts) > 1) {
                $pageInfo = "📄 (" . ($i + 1) . "/" . count($parts) . ")\n\n";
                $part = $pageInfo . $part;
            }
            
            $results[] = self::sendMessage($chatId, $part, $extra);
            
            if ($i < count($parts) - 1) {
                usleep(100000); // 100ms تأخير
            }
        }
        
        return $results;
    }
    
    /**
     * تقسيم نص طويل إلى أجزاء
     * 
     * @param string $text النص
     * @param int $maxLength الحد الأقصى
     * @return array
     */
    public static function splitMessage(string $text, int $maxLength = 4000): array
    {
        if (mb_strlen($text) <= $maxLength) {
            return [$text];
        }
        
        $parts = [];
        $lines = explode("\n", $text);
        $currentPart = '';
        
        foreach ($lines as $line) {
            if (mb_strlen($line) > $maxLength) {
                // السطر أطول من الحد، قسّمه
                if ($currentPart !== '') {
                    $parts[] = trim($currentPart);
                    $currentPart = '';
                }
                $chunks = mb_str_split($line, $maxLength);
                foreach ($chunks as $chunk) {
                    $parts[] = $chunk;
                }
                continue;
            }
            
            $test = $currentPart . ($currentPart ? "\n" : '') . $line;
            
            if (mb_strlen($test) > $maxLength) {
                $parts[] = trim($currentPart);
                $currentPart = $line;
            } else {
                $currentPart = $test;
            }
        }
        
        if ($currentPart !== '') {
            $parts[] = trim($currentPart);
        }
        
        return $parts ?: [$text];
    }
    
    /**
     * إرسال صورة
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $photo مسار الصورة أو file_id أو URL
     * @param string $caption الوصف
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function sendPhoto($chatId, string $photo, string $caption = '', array $extra = []): array
    {
        $params = array_merge([
            'chat_id'    => $chatId,
            'photo'      => $photo,
            'caption'    => $caption,
            'parse_mode' => 'HTML',
        ], $extra);
        
        return self::apiCall('sendPhoto', $params);
    }
    
    /**
     * إرسال مستند/ملف
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $document مسار الملف أو file_id
     * @param string $caption الوصف
     * @param string|null $filename اسم الملف
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function sendDocument($chatId, string $document, string $caption = '', ?string $filename = null, array $extra = []): array
    {
        $params = array_merge([
            'chat_id'    => $chatId,
            'document'   => $document,
            'caption'    => $caption,
            'parse_mode' => 'HTML',
        ], $extra);
        
        if ($filename) {
            // إذا كان ملفاً محلياً، استخدم CURLFile
            if (file_exists($document)) {
                $params['document'] = new CURLFile($document, mime_content_type($document), $filename);
            }
        }
        
        return self::apiCall('sendDocument', $params);
    }
    
    /**
     * إرسال فيديو
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $video مسار الفيديو أو file_id
     * @param string $caption الوصف
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function sendVideo($chatId, string $video, string $caption = '', array $extra = []): array
    {
        $params = array_merge([
            'chat_id'    => $chatId,
            'video'      => $video,
            'caption'    => $caption,
            'parse_mode' => 'HTML',
        ], $extra);
        
        return self::apiCall('sendVideo', $params);
    }
    
    /**
     * إرسال صوت
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $audio مسار الصوت أو file_id
     * @param string $caption الوصف
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function sendAudio($chatId, string $audio, string $caption = '', array $extra = []): array
    {
        $params = array_merge([
            'chat_id'    => $chatId,
            'audio'      => $audio,
            'caption'    => $caption,
            'parse_mode' => 'HTML',
        ], $extra);
        
        return self::apiCall('sendAudio', $params);
    }
    
    /**
     * إرسال رسالة صوتية
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $voice مسار الصوت أو file_id
     * @param string $caption الوصف
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function sendVoice($chatId, string $voice, string $caption = '', array $extra = []): array
    {
        $params = array_merge([
            'chat_id'    => $chatId,
            'voice'      => $voice,
            'caption'    => $caption,
            'parse_mode' => 'HTML',
        ], $extra);
        
        return self::apiCall('sendVoice', $params);
    }
    
    /**
     * إرسال ملصق
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $sticker file_id للملصق
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function sendSticker($chatId, string $sticker, array $extra = []): array
    {
        $params = array_merge([
            'chat_id' => $chatId,
            'sticker' => $sticker,
        ], $extra);
        
        return self::apiCall('sendSticker', $params);
    }
    
    /**
     * إرسال GIF/Animation
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $animation file_id أو URL
     * @param string $caption الوصف
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function sendAnimation($chatId, string $animation, string $caption = '', array $extra = []): array
    {
        $params = array_merge([
            'chat_id'    => $chatId,
            'animation'  => $animation,
            'caption'    => $caption,
            'parse_mode' => 'HTML',
        ], $extra);
        
        return self::apiCall('sendAnimation', $params);
    }
    
    /**
     * إعادة توجيه محتوى (نسخ)
     * 
     * @param int|string $chatId معرف المحادثة الهدف
     * @param int|string $fromChatId معرف المحادثة المصدر
     * @param int $messageId معرف الرسالة
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function copyMessage($chatId, $fromChatId, int $messageId, array $extra = []): array
    {
        $params = array_merge([
            'chat_id'      => $chatId,
            'from_chat_id' => $fromChatId,
            'message_id'   => $messageId,
        ], $extra);
        
        return self::apiCall('copyMessage', $params);
    }
    
    /**
     * التحقق من عضوية المستخدم في قناة/مجموعة
     * 
     * @param int|string $chatId معرف القناة/المجموعة
     * @param int $userId معرف المستخدم
     * @return bool
     */
    public static function isChannelMember($chatId, int $userId): bool
    {
        $result = self::apiCall('getChatMember', [
            'chat_id' => $chatId,
            'user_id' => $userId,
        ]);
        
        if (!($result['ok'] ?? false)) {
            return false;
        }
        
        $status = $result['result']['status'] ?? 'left';
        return in_array($status, ['member', 'administrator', 'creator'], true);
    }
    
    /**
     * التحقق من الاشتراك في قناة التفعيل
     * 
     * @param int $userId معرف المستخدم
     * @return bool
     */
    public static function isSubscribedToChannel(int $userId): bool
    {
        $channelId = self::$config['CHANNEL_ID'] ?? null;
        if (!$channelId) {
            return true; // إذا لم تُحدد قناة، يُعتبر مشتركاً
        }
        
        return self::isChannelMember($channelId, $userId);
    }
    
    /**
     * الحصول على معلومات البوت
     * 
     * @return array|null
     */
    public static function getMe(): ?array
    {
        $result = self::apiCall('getMe');
        return ($result['ok'] ?? false) ? $result['result'] : null;
    }
    
    /**
     * تعيين Webhook
     * 
     * @param string $url رابط الـ Webhook
     * @param array $extra معاملات إضافية
     * @return array
     */
    public static function setWebhook(string $url, array $extra = []): array
    {
        $params = array_merge(['url' => $url], $extra);
        return self::apiCall('setWebhook', $params);
    }
    
    /**
     * حذف Webhook
     * 
     * @return array
     */
    public static function deleteWebhook(): array
    {
        return self::apiCall('deleteWebhook');
    }
    
    /**
     * الحصول على معلومات Webhook
     * 
     * @return array
     */
    public static function getWebhookInfo(): array
    {
        return self::apiCall('getWebhookInfo');
    }
    
    /**
     * الرد على Callback Query
     * 
     * @param string $callbackQueryId معرف الـ callback
     * @param string $text النص (اختياري)
     * @param bool $showAlert إظهار تنبيه
     * @return array
     */
    public static function answerCallbackQuery(string $callbackQueryId, string $text = '', bool $showAlert = false): array
    {
        $params = [
            'callback_query_id' => $callbackQueryId,
        ];
        
        if ($text) {
            $params['text'] = $text;
            $params['show_alert'] = $showAlert;
        }
        
        return self::apiCall('answerCallbackQuery', $params);
    }
    
    /**
     * إرسال محتوى حسب النوع
     * 
     * @param int|string $chatId معرف المحادثة
     * @param string $contentType نوع المحتوى
     * @param string $content المحتوى (نص أو file_id)
     * @param string $caption الوصف
     * @return array
     */
    public static function sendByType($chatId, string $contentType, string $content, string $caption = ''): array
    {
        switch ($contentType) {
            case 'text':
                return self::sendMessage($chatId, $content);
            case 'photo':
                return self::sendPhoto($chatId, $content, $caption);
            case 'video':
                return self::sendVideo($chatId, $content, $caption);
            case 'document':
                return self::sendDocument($chatId, $content, $caption);
            case 'audio':
                return self::sendAudio($chatId, $content, $caption);
            case 'voice':
                return self::sendVoice($chatId, $content, $caption);
            case 'sticker':
                return self::sendSticker($chatId, $content);
            case 'animation':
                return self::sendAnimation($chatId, $content, $caption);
            default:
                return self::sendMessage($chatId, $content);
        }
    }
}
