<?php
/**
 * نظام الفحص الآلي للنظام (System Checker)
 * 
 * يقوم بتنفيذ اختبارات شاملة على النظام ويولد تقارير مفصلة
 * 
 * الاختبارات:
 * 1. اتصال قاعدة البيانات
 * 2. استعلامات أساسية
 * 3. توليد الأكواد
 * 4. تفعيل الأكواد (محاكاة)
 * 5. نظام القسائم
 * 6. إرسال رسائل (dry-run)
 * 7. معالجة 429
 * 8. فحص الاشتراك بالقناة
 */

class SystemChecker
{
    private array $results = [];
    private int $totalTests = 0;
    private int $passedTests = 0;
    private int $failedTests = 0;
    private int $warnings = 0;
    private float $startTime;
    private string $checkType;
    private ?int $triggeredByAdminId;
    private string $triggeredBy;
    
    // مجلد التقارير
    const REPORTS_DIR = __DIR__ . '/../reports/system_checks';
    
    /**
     * تشغيل الفحص الكامل
     */
    public static function runFull(?int $adminId = null, string $triggeredBy = 'admin'): array
    {
        $checker = new self('full', $adminId, $triggeredBy);
        return $checker->run([
            'db_connection',
            'db_queries',
            'code_generation',
            'code_activation',
            'voucher_system',
            'telegram_api',
            'flood_control',
            'channel_subscription',
            'limits_system',
            'cron_status',
        ]);
    }
    
    /**
     * تشغيل فحص سريع
     */
    public static function runQuick(?int $adminId = null, string $triggeredBy = 'admin'): array
    {
        $checker = new self('quick', $adminId, $triggeredBy);
        return $checker->run([
            'db_connection',
            'db_queries',
            'telegram_api',
        ]);
    }
    
    /**
     * تشغيل اختبارات محددة
     */
    public static function runCustom(array $tests, ?int $adminId = null, string $triggeredBy = 'admin'): array
    {
        $checker = new self('custom', $adminId, $triggeredBy);
        return $checker->run($tests);
    }
    
    private function __construct(string $checkType, ?int $adminId, string $triggeredBy)
    {
        $this->checkType = $checkType;
        $this->triggeredByAdminId = $adminId;
        $this->triggeredBy = $triggeredBy;
        $this->startTime = microtime(true);
        
        // إنشاء مجلد التقارير إذا لم يكن موجوداً
        if (!is_dir(self::REPORTS_DIR)) {
            mkdir(self::REPORTS_DIR, 0755, true);
        }
    }
    
    /**
     * تنفيذ الاختبارات
     */
    private function run(array $tests): array
    {
        foreach ($tests as $test) {
            $method = 'test' . str_replace('_', '', ucwords($test, '_'));
            if (method_exists($this, $method)) {
                $this->$method();
            } else {
                $this->addResult($test, 'skip', "اختبار غير معروف: {$test}");
            }
        }
        
        return $this->generateReport();
    }
    
    /**
     * إضافة نتيجة اختبار
     */
    private function addResult(string $test, string $status, string $message, array $details = [], ?string $recommendation = null): void
    {
        $this->totalTests++;
        
        if ($status === 'pass') {
            $this->passedTests++;
        } elseif ($status === 'fail') {
            $this->failedTests++;
        } elseif ($status === 'warn') {
            $this->warnings++;
        }
        
        $this->results[] = [
            'test' => $test,
            'status' => $status,
            'message' => $message,
            'details' => $details,
            'recommendation' => $recommendation,
            'timestamp' => date('Y-m-d H:i:s'),
        ];
    }
    
    // ================================================================
    // الاختبارات
    // ================================================================
    
    /**
     * اختبار اتصال قاعدة البيانات
     */
    private function testDbConnection(): void
    {
        try {
            // محاولة جلب PDO
            if (method_exists('Db', 'getPdo')) {
                $pdo = Db::getPdo();
                if ($pdo) {
                    $result = $pdo->query('SELECT 1');
                    if ($result) {
                        $this->addResult('db_connection', 'pass', 'اتصال قاعدة البيانات يعمل بشكل صحيح');
                        return;
                    }
                }
            }
            
            // محاولة بديلة عبر Db::fetchValue
            $result = Db::fetchValue('SELECT 1');
            if ($result !== false) {
                $this->addResult('db_connection', 'pass', 'اتصال قاعدة البيانات يعمل بشكل صحيح');
            } else {
                $this->addResult('db_connection', 'fail', 'فشل الاتصال بقاعدة البيانات', [],
                    'تحقق من إعدادات قاعدة البيانات في config/env.php');
            }
        } catch (Throwable $e) {
            $this->addResult('db_connection', 'fail', 'فشل الاتصال بقاعدة البيانات', [
                'error' => $e->getMessage(),
            ], 'تحقق من إعدادات قاعدة البيانات في config/env.php');
        }
    }
    
    /**
     * اختبار استعلامات أساسية
     */
    private function testDbQueries(): void
    {
        $tables = [
            'users' => 'جدول المستخدمين',
            'super_distributors' => 'جدول الموزعين',
            'accounts' => 'جدول الحسابات',
            'activation_codes' => 'جدول أكواد التفعيل',
            'user_accounts' => 'جدول اشتراكات المستخدمين',
        ];
        
        $counts = [];
        $errors = [];
        
        foreach ($tables as $table => $name) {
            try {
                $count = Db::fetchValue("SELECT COUNT(*) FROM {$table}");
                $counts[$table] = (int)$count;
            } catch (Throwable $e) {
                $errors[$table] = $e->getMessage();
            }
        }
        
        if (empty($errors)) {
            $this->addResult('db_queries', 'pass', 'استعلامات قاعدة البيانات تعمل بشكل صحيح', [
                'counts' => $counts,
            ]);
        } else {
            $this->addResult('db_queries', 'fail', 'فشل بعض الاستعلامات', [
                'errors' => $errors,
                'counts' => $counts,
            ], 'تحقق من وجود الجداول المطلوبة وصلاحيات المستخدم');
        }
    }
    
    /**
     * اختبار توليد الأكواد
     */
    private function testCodeGeneration(): void
    {
        try {
            // توليد كود تفعيل
            $activationCode = CodeGenerator::generateActivationCode();
            // صيغة مرنة: PREFIX-XXXX-XXXX (أي بادئة + جزئين)
            $validActivation = preg_match('/^[A-Z]+-[A-Z0-9]{4}-[A-Z0-9]{4}$/i', $activationCode);
            
            // توليد كود تحقق (يحتاج secret key)
            $verificationCode = CodeGenerator::generateVerificationCode('test_secret_key');
            // كود التحقق قد يكون 6 أرقام أو أكثر
            $validVerification = strlen($verificationCode) >= 6 && preg_match('/^[0-9]+$/', $verificationCode);
            
            if ($validActivation && $validVerification) {
                $this->addResult('code_generation', 'pass', 'توليد الأكواد يعمل بشكل صحيح', [
                    'sample_activation_code' => $activationCode,
                    'sample_verification_code' => $verificationCode,
                ]);
            } else {
                // فحص تفصيلي
                $details = [
                    'activation_code' => $activationCode,
                    'activation_valid' => (bool)$validActivation,
                    'verification_code' => $verificationCode,
                    'verification_valid' => (bool)$validVerification,
                ];
                
                if (!$validActivation && !$validVerification) {
                    $this->addResult('code_generation', 'fail', 'صيغة الأكواد غير صحيحة', $details,
                        'تحقق من إعدادات CodeGenerator في config');
                } elseif (!$validActivation) {
                    $this->addResult('code_generation', 'warn', 'صيغة كود التفعيل غير قياسية', $details,
                        'الكود يعمل لكن الصيغة قد تختلف عن المتوقعة');
                } else {
                    $this->addResult('code_generation', 'warn', 'صيغة كود التحقق غير قياسية', $details);
                }
            }
        } catch (Throwable $e) {
            $this->addResult('code_generation', 'fail', 'فشل توليد الأكواد', [
                'error' => $e->getMessage(),
            ], 'تحقق من تهيئة CodeGenerator');
        }
    }
    
    /**
     * اختبار تفعيل الأكواد (محاكاة)
     */
    private function testCodeActivation(): void
    {
        try {
            // التحقق من وجود كود غير مستخدم للاختبار
            $unusedCode = Db::fetchOne(
                "SELECT * FROM activation_codes WHERE status = 'unused' LIMIT 1"
            );
            
            if ($unusedCode) {
                // التحقق من منطق التفعيل بدون تنفيذ فعلي
                $account = Db::fetchOne("SELECT * FROM accounts WHERE id = ?", [$unusedCode['account_id']]);
                
                if ($account) {
                    $this->addResult('code_activation', 'pass', 'نظام تفعيل الأكواد جاهز للعمل', [
                        'sample_code' => substr($unusedCode['code'], 0, 8) . '****',
                        'account' => $account['name'],
                        'note' => 'تم التحقق بدون تنفيذ فعلي',
                    ]);
                } else {
                    $this->addResult('code_activation', 'warn', 'كود موجود لكن الحساب غير موجود', [
                        'account_id' => $unusedCode['account_id'],
                    ], 'تحقق من سلامة البيانات');
                }
            } else {
                $this->addResult('code_activation', 'warn', 'لا توجد أكواد غير مستخدمة للاختبار', [], 
                    'هذا طبيعي إذا كان النظام جديداً');
            }
        } catch (Throwable $e) {
            $this->addResult('code_activation', 'fail', 'فشل اختبار تفعيل الأكواد', [
                'error' => $e->getMessage(),
            ]);
        }
    }
    
    /**
     * اختبار نظام القسائم
     */
    private function testVoucherSystem(): void
    {
        try {
            // التحقق من وجود جدول القسائم
            $tableExists = Db::fetchValue(
                "SELECT COUNT(*) FROM information_schema.tables 
                 WHERE table_schema = DATABASE() AND table_name = 'vouchers'"
            );
            
            if (!$tableExists) {
                $this->addResult('voucher_system', 'fail', 'جدول القسائم غير موجود', [],
                    'قم بتشغيل migration 001_vouchers_and_permissions.sql');
                return;
            }
            
            // اختبار إنشاء قسيمة (dry-run)
            $testPayload = [
                'accounts' => [
                    ['account_id' => 1, 'duration_days' => 30]
                ]
            ];
            
            // التحقق من صحة الـ payload فقط
            if (class_exists('Voucher')) {
                $this->addResult('voucher_system', 'pass', 'نظام القسائم جاهز للعمل', [
                    'table_exists' => true,
                    'voucher_class_loaded' => true,
                ]);
            } else {
                $this->addResult('voucher_system', 'warn', 'جدول القسائم موجود لكن الـ class غير محمل', [],
                    'تأكد من تحميل Voucher.php في bootstrap');
            }
            
        } catch (Throwable $e) {
            $this->addResult('voucher_system', 'fail', 'فشل اختبار نظام القسائم', [
                'error' => $e->getMessage(),
            ]);
        }
    }
    
    /**
     * اختبار Telegram API
     */
    private function testTelegramApi(): void
    {
        try {
            // اختبار getMe
            $botInfo = Telegram::getMe();
            
            if ($botInfo && isset($botInfo['id'])) {
                $this->addResult('telegram_api', 'pass', 'اتصال Telegram API يعمل بشكل صحيح', [
                    'bot_id' => $botInfo['id'],
                    'bot_username' => $botInfo['username'] ?? 'N/A',
                    'bot_name' => $botInfo['first_name'] ?? 'N/A',
                ]);
            } else {
                $this->addResult('telegram_api', 'fail', 'فشل الحصول على معلومات البوت', [],
                    'تحقق من صحة TOKEN في config');
            }
        } catch (Throwable $e) {
            $this->addResult('telegram_api', 'fail', 'فشل الاتصال بـ Telegram API', [
                'error' => $e->getMessage(),
            ], 'تحقق من الاتصال بالإنترنت وصحة TOKEN');
        }
    }
    
    /**
     * اختبار معالجة Flood Control (429)
     */
    private function testFloodControl(): void
    {
        try {
            // التحقق من وجود منطق إعادة المحاولة في Telegram class
            $telegramFile = file_get_contents(__DIR__ . '/Telegram.php');
            
            $has429Handling = strpos($telegramFile, '429') !== false;
            $hasRetryAfter = strpos($telegramFile, 'retry_after') !== false;
            $hasSleep = strpos($telegramFile, 'sleep') !== false;
            
            if ($has429Handling && $hasRetryAfter && $hasSleep) {
                $this->addResult('flood_control', 'pass', 'معالجة Flood Control (429) موجودة', [
                    'has_429_check' => true,
                    'has_retry_after' => true,
                    'has_sleep' => true,
                ]);
            } else {
                $this->addResult('flood_control', 'warn', 'معالجة 429 قد تكون غير مكتملة', [
                    'has_429_check' => $has429Handling,
                    'has_retry_after' => $hasRetryAfter,
                    'has_sleep' => $hasSleep,
                ], 'تأكد من وجود منطق إعادة المحاولة عند 429');
            }
        } catch (Throwable $e) {
            $this->addResult('flood_control', 'fail', 'فشل فحص معالجة 429', [
                'error' => $e->getMessage(),
            ]);
        }
    }
    
    /**
     * اختبار فحص الاشتراك بالقناة
     */
    private function testChannelSubscription(): void
    {
        try {
            $channelId = config('required_channel_id');
            
            if (empty($channelId)) {
                $this->addResult('channel_subscription', 'warn', 'لم يتم تحديد قناة إلزامية', [],
                    'يمكنك تحديد REQUIRED_CHANNEL_ID في config');
                return;
            }
            
            // التحقق من أن البوت عضو في القناة
            $botInfo = Telegram::getMe();
            if ($botInfo) {
                $member = Telegram::getChatMember($channelId, $botInfo['id']);
                
                if ($member && in_array($member['status'], ['administrator', 'creator'])) {
                    $this->addResult('channel_subscription', 'pass', 'البوت مسؤول في القناة المطلوبة', [
                        'channel_id' => $channelId,
                        'bot_status' => $member['status'],
                    ]);
                } else {
                    $this->addResult('channel_subscription', 'warn', 'البوت ليس مسؤولاً في القناة', [
                        'channel_id' => $channelId,
                        'bot_status' => $member['status'] ?? 'unknown',
                    ], 'أضف البوت كمسؤول في القناة');
                }
            }
        } catch (Throwable $e) {
            $this->addResult('channel_subscription', 'fail', 'فشل فحص الاشتراك بالقناة', [
                'error' => $e->getMessage(),
            ]);
        }
    }
    
    /**
     * اختبار نظام الحدود
     */
    private function testLimitsSystem(): void
    {
        try {
            // جلب الحدود العامة
            $globalLimits = Limits::getGlobalLimits();
            
            // التحقق من وجود الحدود
            if ($globalLimits) {
                $this->addResult('limits_system', 'pass', 'نظام الحدود يعمل بشكل صحيح', [
                    'global_limits' => $globalLimits,
                ]);
            } else {
                $this->addResult('limits_system', 'warn', 'لا توجد حدود عامة محددة', [],
                    'قم بتعيين الحدود العامة من لوحة الإدارة');
            }
        } catch (Throwable $e) {
            $this->addResult('limits_system', 'fail', 'فشل اختبار نظام الحدود', [
                'error' => $e->getMessage(),
            ]);
        }
    }
    
    /**
     * اختبار حالة Cron Jobs
     */
    private function testCronStatus(): void
    {
        try {
            // التحقق من آخر تشغيل للـ cron jobs
            $lastDailyReset = Db::fetchValue(
                "SELECT MAX(last_daily_reset) FROM superdist_accounts"
            );
            
            $lastEvent = Db::fetchOne(
                "SELECT * FROM event_logs WHERE event_type LIKE 'cron_%' ORDER BY created_at DESC LIMIT 1"
            );
            
            $details = [
                'last_daily_reset' => $lastDailyReset,
                'last_cron_event' => $lastEvent ? $lastEvent['created_at'] : null,
            ];
            
            // التحقق من أن الـ cron تعمل
            if ($lastDailyReset && date('Y-m-d', strtotime($lastDailyReset)) === date('Y-m-d')) {
                $this->addResult('cron_status', 'pass', 'Cron Jobs تعمل بشكل صحيح', $details);
            } elseif ($lastDailyReset) {
                $this->addResult('cron_status', 'warn', 'Cron Jobs لم تعمل اليوم', $details,
                    'تحقق من إعدادات crontab');
            } else {
                $this->addResult('cron_status', 'warn', 'لا توجد سجلات تشغيل Cron', $details,
                    'قم بإعداد crontab لتشغيل الملفات في مجلد cron/');
            }
        } catch (Throwable $e) {
            $this->addResult('cron_status', 'fail', 'فشل فحص حالة Cron', [
                'error' => $e->getMessage(),
            ]);
        }
    }
    
    // ================================================================
    // توليد التقرير
    // ================================================================
    
    /**
     * توليد التقرير النهائي
     */
    private function generateReport(): array
    {
        $duration = round((microtime(true) - $this->startTime) * 1000);
        $timestamp = date('Y-m-d_H-i-s');
        
        // الملخص
        $summary = $this->generateSummary();
        
        // التوصيات
        $recommendations = $this->extractRecommendations();
        
        // بناء التقرير
        $report = [
            'check_type' => $this->checkType,
            'triggered_by' => $this->triggeredBy,
            'triggered_by_admin_id' => $this->triggeredByAdminId,
            'timestamp' => date('Y-m-d H:i:s'),
            'duration_ms' => $duration,
            'stats' => [
                'total' => $this->totalTests,
                'passed' => $this->passedTests,
                'failed' => $this->failedTests,
                'warnings' => $this->warnings,
            ],
            'summary' => $summary,
            'results' => $this->results,
            'recommendations' => $recommendations,
        ];
        
        // حفظ التقارير
        $jsonFile = self::REPORTS_DIR . "/{$timestamp}.json";
        $txtFile = self::REPORTS_DIR . "/{$timestamp}.txt";
        
        file_put_contents($jsonFile, json_encode($report, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        file_put_contents($txtFile, $this->generateTextReport($report));
        
        $report['report_file_json'] = $jsonFile;
        $report['report_file_txt'] = $txtFile;
        
        // حفظ في قاعدة البيانات
        $this->saveToDatabase($report);
        
        return $report;
    }
    
    /**
     * توليد ملخص نصي
     */
    private function generateSummary(): string
    {
        $status = $this->failedTests === 0 ? '✅ جميع الاختبارات نجحت' : '❌ يوجد اختبارات فاشلة';
        
        return sprintf(
            "%s\n\n📊 الإحصائيات:\n- إجمالي: %d\n- ناجح: %d\n- فاشل: %d\n- تحذيرات: %d",
            $status,
            $this->totalTests,
            $this->passedTests,
            $this->failedTests,
            $this->warnings
        );
    }
    
    /**
     * استخراج التوصيات
     */
    private function extractRecommendations(): array
    {
        $recommendations = [];
        foreach ($this->results as $result) {
            if (!empty($result['recommendation'])) {
                $recommendations[] = [
                    'test' => $result['test'],
                    'status' => $result['status'],
                    'recommendation' => $result['recommendation'],
                ];
            }
        }
        return $recommendations;
    }
    
    /**
     * توليد تقرير نصي
     */
    private function generateTextReport(array $report): string
    {
        $text = "═══════════════════════════════════════════\n";
        $text .= "     تقرير فحص النظام - System Check Report\n";
        $text .= "═══════════════════════════════════════════\n\n";
        
        $text .= "📅 التاريخ: {$report['timestamp']}\n";
        $text .= "⏱️ المدة: {$report['duration_ms']}ms\n";
        $text .= "🔧 نوع الفحص: {$report['check_type']}\n";
        $text .= "👤 المشغّل: {$report['triggered_by']}\n\n";
        
        $text .= "───────────────────────────────────────────\n";
        $text .= "                  النتائج\n";
        $text .= "───────────────────────────────────────────\n\n";
        
        foreach ($report['results'] as $result) {
            $icon = match($result['status']) {
                'pass' => '✅',
                'fail' => '❌',
                'warn' => '⚠️',
                default => '⏭️',
            };
            
            $text .= "{$icon} [{$result['test']}]\n";
            $text .= "   {$result['message']}\n";
            
            if (!empty($result['recommendation'])) {
                $text .= "   💡 {$result['recommendation']}\n";
            }
            $text .= "\n";
        }
        
        $text .= "───────────────────────────────────────────\n";
        $text .= "                  الملخص\n";
        $text .= "───────────────────────────────────────────\n\n";
        $text .= $report['summary'] . "\n\n";
        
        $text .= "═══════════════════════════════════════════\n";
        
        return $text;
    }
    
    /**
     * حفظ التقرير في قاعدة البيانات
     */
    private function saveToDatabase(array $report): void
    {
        try {
            Db::insert('system_check_reports', [
                'check_type' => $report['check_type'],
                'triggered_by' => $report['triggered_by'],
                'triggered_by_admin_id' => $report['triggered_by_admin_id'],
                'total_tests' => $report['stats']['total'],
                'passed_tests' => $report['stats']['passed'],
                'failed_tests' => $report['stats']['failed'],
                'warnings' => $report['stats']['warnings'],
                'results' => json_encode($report['results'], JSON_UNESCAPED_UNICODE),
                'summary' => $report['summary'],
                'recommendations' => json_encode($report['recommendations'], JSON_UNESCAPED_UNICODE),
                'report_file_json' => $report['report_file_json'] ?? null,
                'report_file_txt' => $report['report_file_txt'] ?? null,
                'duration_ms' => $report['duration_ms'],
            ]);
        } catch (Throwable $e) {
            // تجاهل الخطأ - الجدول قد لا يكون موجوداً بعد
            Logger::error('Failed to save system check report to DB', ['error' => $e->getMessage()]);
        }
    }
    
    /**
     * جلب التقارير السابقة
     */
    public static function getReports(int $limit = 20): array
    {
        try {
            return Db::fetchAll(
                "SELECT * FROM system_check_reports ORDER BY created_at DESC LIMIT ?",
                [$limit]
            );
        } catch (Throwable $e) {
            return [];
        }
    }
    
    /**
     * جلب تقرير محدد
     */
    public static function getReport(int $id): ?array
    {
        try {
            return Db::fetchOne(
                "SELECT * FROM system_check_reports WHERE id = ?",
                [$id]
            );
        } catch (Throwable $e) {
            return null;
        }
    }
}
