<?php
/**
 * ============================================================
 * مكتبة تسجيل الأحداث (Logger)
 * ============================================================
 * 
 * تسجّل الأحداث المهمة في:
 * - قاعدة البيانات (جدول event_logs)
 * - ملفات السجلات (logs/*.log)
 * 
 * الاستخدام:
 *   Logger::init($config);
 *   Logger::event('user_registered', 'user', $userId, 'user', $userId, ['username' => '@user']);
 *   Logger::error('Database connection failed', ['error' => $e->getMessage()]);
 */

class Logger
{
    /** @var array إعدادات النظام */
    private static array $config = [];
    
    /** @var string مسار ملفات السجلات */
    private static string $logPath = '';
    
    /** @var bool وضع التصحيح */
    private static bool $debugMode = false;
    
    /**
     * تهيئة المكتبة
     * 
     * @param array $config إعدادات من env.php
     */
    public static function init(array $config): void
    {
        self::$config = $config;
        self::$logPath = rtrim($config['LOG_PATH'] ?? __DIR__ . '/../logs/', '/') . '/';
        self::$debugMode = $config['DEBUG_MODE'] ?? false;
        
        // إنشاء مجلد السجلات إذا لم يكن موجوداً
        if (!is_dir(self::$logPath)) {
            mkdir(self::$logPath, 0755, true);
        }
    }
    
    /**
     * تسجيل حدث في قاعدة البيانات
     * 
     * @param string $eventType نوع الحدث
     * @param string $actorType نوع الفاعل (admin, superdist, user, system)
     * @param string|int|null $actorId معرف الفاعل
     * @param string|null $targetType نوع الهدف
     * @param string|int|null $targetId معرف الهدف
     * @param array|null $details تفاصيل إضافية
     * @return int|null معرف السجل
     */
    public static function event(
        string $eventType,
        string $actorType,
        $actorId = null,
        ?string $targetType = null,
        $targetId = null,
        ?array $details = null
    ): ?int {
        try {
            $id = Db::insert('event_logs', [
                'event_type'  => $eventType,
                'actor_type'  => $actorType,
                'actor_id'    => $actorId !== null ? (string) $actorId : null,
                'target_type' => $targetType,
                'target_id'   => $targetId !== null ? (string) $targetId : null,
                'details'     => $details !== null ? json_encode($details, JSON_UNESCAPED_UNICODE) : null,
                'ip_address'  => self::getClientIp(),
            ]);
            
            // سجّل في الملف أيضاً إذا كان وضع التصحيح مفعّلاً
            if (self::$debugMode) {
                self::toFile('events', "[{$eventType}] actor:{$actorType}:{$actorId} target:{$targetType}:{$targetId}");
            }
            
            return (int) $id;
        } catch (Throwable $e) {
            self::error('Failed to log event', [
                'event_type' => $eventType,
                'error'      => $e->getMessage(),
            ]);
            return null;
        }
    }
    
    /**
     * تسجيل خطأ
     * 
     * @param string $message رسالة الخطأ
     * @param array $context سياق إضافي
     */
    public static function error(string $message, array $context = []): void
    {
        $logEntry = self::formatLogEntry('ERROR', $message, $context);
        self::toFile('error', $logEntry);
        
        // سجّل في قاعدة البيانات أيضاً
        try {
            Db::insert('event_logs', [
                'event_type'  => 'error',
                'actor_type'  => 'system',
                'details'     => json_encode(['message' => $message, 'context' => $context], JSON_UNESCAPED_UNICODE),
                'ip_address'  => self::getClientIp(),
            ]);
        } catch (Throwable $e) {
            // تجاهل - لا نريد حلقة لانهائية
        }
    }
    
    /**
     * تسجيل تحذير
     * 
     * @param string $message رسالة التحذير
     * @param array $context سياق إضافي
     */
    public static function warning(string $message, array $context = []): void
    {
        $logEntry = self::formatLogEntry('WARNING', $message, $context);
        self::toFile('warning', $logEntry);
    }
    
    /**
     * تسجيل معلومات
     * 
     * @param string $message الرسالة
     * @param array $context سياق إضافي
     */
    public static function info(string $message, array $context = []): void
    {
        if (!self::$debugMode) {
            return;
        }
        
        $logEntry = self::formatLogEntry('INFO', $message, $context);
        self::toFile('info', $logEntry);
    }
    
    /**
     * تسجيل تصحيح (debug)
     * 
     * @param string $message الرسالة
     * @param array $context سياق إضافي
     */
    public static function debug(string $message, array $context = []): void
    {
        if (!self::$debugMode) {
            return;
        }
        
        $logEntry = self::formatLogEntry('DEBUG', $message, $context);
        self::toFile('debug', $logEntry);
    }
    
    // ================================================================
    // أحداث محددة مسبقاً
    // ================================================================
    
    /**
     * تسجيل تفعيل حساب
     * 
     * @param int $userId
     * @param int $accountId
     * @param string $code
     * @param int|null $superdistId
     */
    public static function accountActivated(int $userId, int $accountId, string $code, ?int $superdistId = null): void
    {
        self::event('code_activated', 'user', $userId, 'account', $accountId, [
            'code'         => $code,
            'superdist_id' => $superdistId,
        ]);
    }
    
    /**
     * تسجيل طلب كود تحقق
     * 
     * @param int $userId
     * @param int $accountId
     * @param string $code
     */
    public static function verificationRequested(int $userId, int $accountId, string $code): void
    {
        self::event('verification_requested', 'user', $userId, 'account', $accountId, [
            'code' => $code,
        ]);
    }
    
    /**
     * تسجيل توليد أكواد تفعيل
     * 
     * @param int $superdistId
     * @param int $accountId
     * @param int $count
     * @param array $codes
     */
    public static function codesGenerated(int $superdistId, int $accountId, int $count, array $codes = []): void
    {
        self::event('code_generated', 'superdist', $superdistId, 'account', $accountId, [
            'count' => $count,
            'codes' => array_slice($codes, 0, 10), // أول 10 فقط
        ]);
    }
    
    /**
     * تسجيل حظر مستخدم
     * 
     * @param int $userId
     * @param string $reason
     * @param string $bannedBy
     */
    public static function userBanned(int $userId, string $reason, string $bannedBy): void
    {
        self::event('user_banned', 'admin', $bannedBy, 'user', $userId, [
            'reason' => $reason,
        ]);
    }
    
    /**
     * تسجيل فك حظر مستخدم
     * 
     * @param int $userId
     * @param string $unbannedBy
     */
    public static function userUnbanned(int $userId, string $unbannedBy): void
    {
        self::event('user_unbanned', 'admin', $unbannedBy, 'user', $userId);
    }
    
    /**
     * تسجيل بث
     * 
     * @param string $broadcasterType
     * @param string $broadcasterId
     * @param string $audienceType
     * @param int $total
     * @param int $success
     * @param int $failed
     */
    public static function broadcastCompleted(
        string $broadcasterType,
        string $broadcasterId,
        string $audienceType,
        int $total,
        int $success,
        int $failed
    ): void {
        self::event('broadcast_completed', $broadcasterType, $broadcasterId, null, null, [
            'audience_type' => $audienceType,
            'total'         => $total,
            'success'       => $success,
            'failed'        => $failed,
        ]);
    }
    
    /**
     * تسجيل تجاوز حد
     * 
     * @param int $userId
     * @param int $accountId
     * @param string $limitType
     * @param int $limit
     * @param int $used
     */
    public static function limitExceeded(int $userId, int $accountId, string $limitType, int $limit, int $used): void
    {
        self::event('limit_exceeded', 'user', $userId, 'account', $accountId, [
            'limit_type' => $limitType,
            'limit'      => $limit,
            'used'       => $used,
        ]);
    }
    
    // ================================================================
    // دوال مساعدة
    // ================================================================
    
    /**
     * تنسيق سجل للملف
     * 
     * @param string $level مستوى السجل
     * @param string $message الرسالة
     * @param array $context السياق
     * @return string
     */
    private static function formatLogEntry(string $level, string $message, array $context = []): string
    {
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = $context ? ' ' . json_encode($context, JSON_UNESCAPED_UNICODE) : '';
        
        return "[{$timestamp}] [{$level}] {$message}{$contextStr}";
    }
    
    /**
     * الكتابة في ملف سجل
     * 
     * @param string $filename اسم الملف (بدون امتداد)
     * @param string $entry السجل
     */
    private static function toFile(string $filename, string $entry): void
    {
        $filepath = self::$logPath . $filename . '_' . date('Y-m-d') . '.log';
        
        file_put_contents($filepath, $entry . PHP_EOL, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * الحصول على عنوان IP للعميل
     * 
     * @return string|null
     */
    private static function getClientIp(): ?string
    {
        $headers = [
            'HTTP_CF_CONNECTING_IP', // Cloudflare
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR',
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                // في حالة X-Forwarded-For قد يكون هناك عدة IPs
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                return $ip;
            }
        }
        
        return null;
    }
    
    /**
     * جلب آخر الأحداث من قاعدة البيانات
     * 
     * @param int $limit
     * @param string|null $eventType فلتر بنوع الحدث
     * @return array
     */
    public static function getRecentEvents(int $limit = 50, ?string $eventType = null): array
    {
        if ($eventType) {
            return Db::fetchAll(
                "SELECT * FROM event_logs WHERE event_type = ? ORDER BY created_at DESC LIMIT ?",
                [$eventType, $limit]
            );
        }
        
        return Db::fetchAll(
            "SELECT * FROM event_logs ORDER BY created_at DESC LIMIT ?",
            [$limit]
        );
    }
}
