<?php
/**
 * ============================================================
 * طبقة قاعدة البيانات (Database Layer)
 * ============================================================
 * 
 * توفر اتصالاً آمناً بقاعدة البيانات عبر PDO.
 * تستخدم نمط Singleton لضمان اتصال واحد فقط.
 * 
 * الاستخدام:
 *   $db = Db::getInstance();
 *   $users = $db->fetchAll("SELECT * FROM users WHERE status = ?", ['active']);
 */

class Db
{
    /** @var PDO|null */
    private static ?PDO $instance = null;
    
    /** @var array إعدادات الاتصال */
    private static array $config = [];
    
    /**
     * منع إنشاء نسخة مباشرة (Singleton)
     */
    private function __construct() {}
    private function __clone() {}
    
    /**
     * تهيئة الإعدادات
     * 
     * @param array $config إعدادات الاتصال من env.php
     */
    public static function init(array $config): void
    {
        self::$config = $config;
    }
    
    /**
     * الحصول على نسخة PDO (Singleton)
     * 
     * @return PDO
     * @throws PDOException
     */
    public static function getInstance(): PDO
    {
        if (self::$instance === null) {
            $dsn = sprintf(
                "mysql:host=%s;dbname=%s;charset=%s",
                self::$config['DB_HOST'] ?? 'localhost',
                self::$config['DB_NAME'] ?? '',
                self::$config['DB_CHARSET'] ?? 'utf8mb4'
            );
            
            self::$instance = new PDO(
                $dsn,
                self::$config['DB_USER'] ?? '',
                self::$config['DB_PASS'] ?? '',
                [
                    PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES   => false,
                    PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
                ]
            );
        }
        
        return self::$instance;
    }
    
    /**
     * تنفيذ استعلام وإرجاع جميع النتائج
     * 
     * @param string $sql الاستعلام
     * @param array $params المعاملات
     * @return array
     */
    public static function fetchAll(string $sql, array $params = []): array
    {
        $stmt = self::getInstance()->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }
    
    /**
     * تنفيذ استعلام وإرجاع صف واحد
     * 
     * @param string $sql الاستعلام
     * @param array $params المعاملات
     * @return array|null
     */
    public static function fetchOne(string $sql, array $params = []): ?array
    {
        $stmt = self::getInstance()->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch();
        return $result ?: null;
    }
    
    /**
     * تنفيذ استعلام وإرجاع قيمة واحدة
     * 
     * @param string $sql الاستعلام
     * @param array $params المعاملات
     * @return mixed
     */
    public static function fetchValue(string $sql, array $params = [])
    {
        $stmt = self::getInstance()->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchColumn();
    }
    
    /**
     * تنفيذ استعلام وإرجاع عمود واحد كمصفوفة
     * 
     * @param string $sql الاستعلام
     * @param array $params المعاملات
     * @return array
     */
    public static function fetchColumn(string $sql, array $params = []): array
    {
        $stmt = self::getInstance()->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
    
    /**
     * تنفيذ استعلام (INSERT, UPDATE, DELETE)
     * 
     * @param string $sql الاستعلام
     * @param array $params المعاملات
     * @return int عدد الصفوف المتأثرة
     */
    public static function execute(string $sql, array $params = []): int
    {
        $stmt = self::getInstance()->prepare($sql);
        $stmt->execute($params);
        return $stmt->rowCount();
    }
    
    /**
     * إدراج صف وإرجاع المعرف الجديد
     * 
     * @param string $table اسم الجدول
     * @param array $data البيانات [column => value]
     * @return int|string المعرف الجديد
     */
    public static function insert(string $table, array $data)
    {
        $columns = implode(', ', array_map(fn($col) => "`{$col}`", array_keys($data)));
        $placeholders = implode(', ', array_fill(0, count($data), '?'));
        
        $sql = "INSERT INTO `{$table}` ({$columns}) VALUES ({$placeholders})";
        
        $stmt = self::getInstance()->prepare($sql);
        $stmt->execute(array_values($data));
        
        return self::getInstance()->lastInsertId();
    }
    
    /**
     * تحديث صفوف
     * 
     * @param string $table اسم الجدول
     * @param array $data البيانات للتحديث [column => value]
     * @param string $where شرط WHERE (بدون كلمة WHERE)
     * @param array $whereParams معاملات الشرط
     * @return int عدد الصفوف المتأثرة
     */
    public static function update(string $table, array $data, string $where, array $whereParams = []): int
    {
        $setParts = array_map(fn($col) => "`{$col}` = ?", array_keys($data));
        $setClause = implode(', ', $setParts);
        
        $sql = "UPDATE `{$table}` SET {$setClause} WHERE {$where}";
        
        $params = array_merge(array_values($data), $whereParams);
        
        $stmt = self::getInstance()->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->rowCount();
    }
    
    /**
     * حذف صفوف
     * 
     * @param string $table اسم الجدول
     * @param string $where شرط WHERE
     * @param array $params معاملات الشرط
     * @return int عدد الصفوف المحذوفة
     */
    public static function delete(string $table, string $where, array $params = []): int
    {
        $sql = "DELETE FROM `{$table}` WHERE {$where}";
        
        $stmt = self::getInstance()->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->rowCount();
    }
    
    /**
     * بدء معاملة (Transaction)
     */
    public static function beginTransaction(): bool
    {
        $pdo = self::getInstance();
        if ($pdo->inTransaction()) {
            return true; // transaction نشط بالفعل
        }
        return $pdo->beginTransaction();
    }
    
    /**
     * التحقق من وجود معاملة نشطة
     */
    public static function inTransaction(): bool
    {
        return self::getInstance()->inTransaction();
    }
    
    /**
     * تأكيد المعاملة
     */
    public static function commit(): bool
    {
        $pdo = self::getInstance();
        if (!$pdo->inTransaction()) {
            return true; // لا توجد معاملة نشطة
        }
        return $pdo->commit();
    }
    
    /**
     * التراجع عن المعاملة
     */
    public static function rollback(): bool
    {
        $pdo = self::getInstance();
        if (!$pdo->inTransaction()) {
            return true; // لا توجد معاملة نشطة
        }
        return $pdo->rollBack();
    }
    
    /**
     * الحصول على آخر معرف مُدرج
     * 
     * @return string
     */
    public static function lastInsertId(): string
    {
        return self::getInstance()->lastInsertId();
    }
    
    /**
     * التحقق من وجود صف
     * 
     * @param string $table اسم الجدول
     * @param string $where شرط WHERE
     * @param array $params المعاملات
     * @return bool
     */
    public static function exists(string $table, string $where, array $params = []): bool
    {
        $sql = "SELECT 1 FROM `{$table}` WHERE {$where} LIMIT 1";
        return self::fetchValue($sql, $params) !== false;
    }
    
    /**
     * عدّ الصفوف
     * 
     * @param string $table اسم الجدول
     * @param string $where شرط WHERE (اختياري)
     * @param array $params المعاملات
     * @return int
     */
    public static function count(string $table, string $where = '1=1', array $params = []): int
    {
        $sql = "SELECT COUNT(*) FROM `{$table}` WHERE {$where}";
        return (int) self::fetchValue($sql, $params);
    }
}
