<?php
/**
 * ============================================================
 * مهمة يومية - التعامل مع انتهاء حسابات السوبر موزعين
 * ============================================================
 * 
 * تُنفذ: يومياً عند الساعة 01:00
 * 
 * Cron Command (cPanel):
 *   0 1 * * * /usr/bin/php /home/username/public_html/cron/superdist_expiry.php
 * 
 * الوظائف:
 *   1. تحديث حالة حسابات السوبر موزعين المنتهية
 *   2. تعطيل الأكواد غير المستخدمة للحسابات المنتهية
 *   3. (اختياري) إرسال تنبيهات قبل الانتهاء
 */

if (php_sapi_name() !== 'cli' && !defined('CRON_ALLOWED')) {
    http_response_code(403);
    exit('CLI only');
}

define('BOT_STARTED', true);
define('BOT_TYPE', 'cron');

require __DIR__ . '/../libs/bootstrap.php';

$startTime = microtime(true);
$log = [];

echo "=== Super Distributor Expiry Check ===\n";
echo "Started at: " . date('Y-m-d H:i:s') . "\n\n";

try {
    // ============================================================
    // 1. جلب حسابات السوبر موزعين المنتهية حديثاً
    // ============================================================
    echo "1. Checking expired superdist accounts...\n";
    
    $expiredAccounts = Db::fetchAll(
        "SELECT sa.*, sd.telegram_id, sd.display_name, a.slug, a.name as account_name
         FROM superdist_accounts sa
         JOIN super_distributors sd ON sd.id = sa.superdist_id
         JOIN accounts a ON a.id = sa.account_id
         WHERE sa.end_at < NOW()
           AND sa.end_at > DATE_SUB(NOW(), INTERVAL 1 DAY)
           AND sd.status = 'active'"
    );
    
    echo "   - Found " . count($expiredAccounts) . " newly expired accounts\n";
    
    // ============================================================
    // 2. تعطيل الأكواد غير المستخدمة للحسابات المنتهية
    // ============================================================
    echo "2. Disabling unused codes for expired accounts...\n";
    
    $totalDisabled = 0;
    
    foreach ($expiredAccounts as $ea) {
        $affected = Db::execute(
            "UPDATE activation_codes 
             SET status = 'expired' 
             WHERE superdist_id = ? 
               AND account_id = ? 
               AND status = 'unused'",
            [$ea['superdist_id'], $ea['account_id']]
        );
        
        $totalDisabled += $affected;
        
        if ($affected > 0) {
            echo "   - Superdist {$ea['superdist_id']}, Account {$ea['account_id']}: {$affected} codes expired\n";
            
            // تسجيل الحدث
            Logger::event('superdist_account_expired', 'system', null, 'superdist_account', $ea['id'], [
                'superdist_id' => $ea['superdist_id'],
                'account_id'   => $ea['account_id'],
                'codes_expired'=> $affected,
            ]);
        }
    }
    
    echo "   - Total disabled: {$totalDisabled} codes\n";
    $log[] = "Expired accounts: " . count($expiredAccounts);
    $log[] = "Disabled codes: {$totalDisabled}";
    
    // ============================================================
    // 3. إرسال تنبيهات قبل الانتهاء (اختياري)
    // ============================================================
    echo "3. Sending expiry warnings (3 days before)...\n";
    
    $expiringAccounts = Db::fetchAll(
        "SELECT sa.*, sd.telegram_id, sd.display_name, a.slug, a.name as account_name
         FROM superdist_accounts sa
         JOIN super_distributors sd ON sd.id = sa.superdist_id
         JOIN accounts a ON a.id = sa.account_id
         WHERE sa.end_at BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL 3 DAY)
           AND sd.status = 'active'"
    );
    
    echo "   - Found " . count($expiringAccounts) . " accounts expiring soon\n";
    
    // يمكن إضافة إرسال تنبيهات هنا
    // foreach ($expiringAccounts as $ea) {
    //     $daysRemaining = ceil((strtotime($ea['end_at']) - time()) / 86400);
    //     Telegram::sendMessage($ea['telegram_id'], "⚠️ تنبيه: حسابك {$ea['account_name']} سينتهي خلال {$daysRemaining} يوم.");
    // }
    
    $log[] = "Expiring soon: " . count($expiringAccounts);
    
    // ============================================================
    // تسجيل الحدث
    // ============================================================
    Logger::event('cron_executed', 'system', null, null, null, [
        'job'     => 'superdist_expiry',
        'results' => $log,
    ]);
    
} catch (Throwable $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    Logger::error('Superdist expiry cron failed', ['error' => $e->getMessage()]);
}

$duration = round((microtime(true) - $startTime) * 1000, 2);
echo "\nCompleted in {$duration}ms\n";
