<?php
/**
 * Verification Run - فحص شامل للنظام بعد التعديلات
 * 
 * يقوم بـ:
 * - فحص اتصال DB
 * - التحقق من schema والأعمدة المطلوبة
 * - فحص توافق الكود
 * - اختبارات وظيفية (functional tests)
 * - توليد تقرير نهائي
 */

// التحقق من CLI
if (php_sapi_name() !== 'cli') {
    die("This script can only be run from command line\n");
}

// تعريف الثوابت
define('BOT_STARTED', true);
define('BOT_TYPE', 'cli');

// إنشاء مجلد التقارير
$reportsDir = __DIR__ . '/../reports/verification';
if (!is_dir($reportsDir)) {
    mkdir($reportsDir, 0755, true);
}

$timestamp = date('Y-m-d_H-i-s');
$startTime = microtime(true);

// ============================================================
// تهيئة التقرير
// ============================================================

$report = [
    'timestamp' => date('Y-m-d H:i:s'),
    'environment' => 'staging',
    'summary' => [],
    'details' => [
        'db_connection' => null,
        'basic_queries' => null,
        'schema_checks' => [],
        'required_columns' => [],
        'code_compatibility' => [],
        'functional_tests' => [],
        'logs_errors' => [],
        'recommendations' => [],
    ],
    'stats' => [
        'total_checks' => 0,
        'passed' => 0,
        'failed' => 0,
        'warnings' => 0,
    ],
];

$textReport = [];
$textReport[] = "═══════════════════════════════════════════════════════════════";
$textReport[] = "           VERIFICATION RUN - System Check Report";
$textReport[] = "═══════════════════════════════════════════════════════════════";
$textReport[] = "";
$textReport[] = "📅 Timestamp: " . date('Y-m-d H:i:s');
$textReport[] = "🔧 Environment: staging";
$textReport[] = "";

// ============================================================
// Helper Functions
// ============================================================

function addCheck(&$report, string $category, string $name, bool $passed, string $message, array $details = [], ?string $recommendation = null): void
{
    $report['stats']['total_checks']++;
    
    if ($passed) {
        $report['stats']['passed']++;
        $status = 'PASS';
    } else {
        $report['stats']['failed']++;
        $status = 'FAIL';
    }
    
    $check = [
        'name' => $name,
        'status' => $status,
        'message' => $message,
        'details' => $details,
        'recommendation' => $recommendation,
    ];
    
    if (is_array($report['details'][$category])) {
        $report['details'][$category][] = $check;
    } else {
        $report['details'][$category] = $check;
    }
    
    if ($recommendation) {
        $report['details']['recommendations'][] = [
            'check' => $name,
            'recommendation' => $recommendation,
        ];
    }
}

function addWarning(&$report, string $name, string $message): void
{
    $report['stats']['warnings']++;
    $report['details']['recommendations'][] = [
        'check' => $name,
        'type' => 'warning',
        'message' => $message,
    ];
}

echo "═══════════════════════════════════════════════════════════════\n";
echo "           VERIFICATION RUN - System Check\n";
echo "═══════════════════════════════════════════════════════════════\n\n";

// ============================================================
// A. تحميل Bootstrap
// ============================================================

echo "📦 Loading bootstrap...\n";

try {
    require __DIR__ . '/../libs/bootstrap.php';
    echo "   ✅ Bootstrap loaded successfully\n\n";
} catch (Throwable $e) {
    echo "   ❌ FATAL: Failed to load bootstrap: " . $e->getMessage() . "\n";
    $report['summary']['bootstrap'] = 'FAIL';
    $report['details']['bootstrap_error'] = $e->getMessage();
    
    // حفظ التقرير الجزئي
    file_put_contents("{$reportsDir}/{$timestamp}_verification.json", json_encode($report, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    exit(1);
}

// ============================================================
// B. فحص اتصال DB
// ============================================================

echo "🔌 B. Database Connection Check\n";
echo "─────────────────────────────────────────────────────────────────\n";

try {
    $result = Db::fetchValue("SELECT 1");
    $passed = ($result == 1);
    
    addCheck($report, 'db_connection', 'DB Connection', $passed, 
        $passed ? 'Database connection successful' : 'Connection returned unexpected value');
    
    echo "   " . ($passed ? "✅" : "❌") . " SELECT 1 = {$result}\n";
    $textReport[] = ($passed ? "✅" : "❌") . " DB Connection: " . ($passed ? "OK" : "FAILED");
    
    $report['summary']['db_connection'] = $passed ? 'PASS' : 'FAIL';
    
} catch (Throwable $e) {
    addCheck($report, 'db_connection', 'DB Connection', false, 
        'Connection failed: ' . $e->getMessage(), [], 'Check database credentials in config/env.php');
    
    echo "   ❌ Connection failed: " . $e->getMessage() . "\n";
    $textReport[] = "❌ DB Connection: FAILED - " . $e->getMessage();
    $report['summary']['db_connection'] = 'FAIL';
}

echo "\n";

// ============================================================
// B2. استعلامات حسابية أساسية
// ============================================================

echo "📊 B2. Basic Count Queries\n";
echo "─────────────────────────────────────────────────────────────────\n";

$countQueries = [
    'accounts' => 'SELECT COUNT(*) FROM accounts',
    'super_distributors' => 'SELECT COUNT(*) FROM super_distributors',
    'users' => 'SELECT COUNT(*) FROM users',
    'activation_codes' => 'SELECT COUNT(*) FROM activation_codes',
    'user_accounts' => 'SELECT COUNT(*) FROM user_accounts',
];

$counts = [];
$allCountsOk = true;

foreach ($countQueries as $table => $query) {
    try {
        $count = Db::fetchValue($query);
        $counts[$table] = (int)$count;
        echo "   ✅ {$table}: {$count}\n";
    } catch (Throwable $e) {
        $counts[$table] = 'ERROR: ' . $e->getMessage();
        $allCountsOk = false;
        echo "   ❌ {$table}: ERROR - " . $e->getMessage() . "\n";
    }
}

addCheck($report, 'basic_queries', 'Basic Count Queries', $allCountsOk,
    $allCountsOk ? 'All count queries successful' : 'Some count queries failed',
    ['counts' => $counts]);

$report['summary']['basic_queries'] = $allCountsOk ? 'PASS' : 'FAIL';
$textReport[] = ($allCountsOk ? "✅" : "❌") . " Basic Queries: " . json_encode($counts);

echo "\n";

// ============================================================
// C. فحص Schema والأعمدة المطلوبة
// ============================================================

echo "🗄️ C. Schema & Required Columns Check\n";
echo "─────────────────────────────────────────────────────────────────\n";

// الأعمدة المطلوبة
$requiredColumns = [
    ['table' => 'super_distributors', 'column' => 'can_broadcast', 'critical' => true],
    ['table' => 'super_distributors', 'column' => 'broadcast_allowed_types', 'critical' => true],
    ['table' => 'user_accounts', 'column' => 'access_expires_at', 'critical' => true],
    ['table' => 'user_accounts', 'column' => 'access_type', 'critical' => true],
    ['table' => 'superdist_accounts', 'column' => 'last_daily_reset', 'critical' => false],
    ['table' => 'superdist_accounts', 'column' => 'last_weekly_reset', 'critical' => false],
    ['table' => 'superdist_accounts', 'column' => 'last_monthly_reset', 'critical' => false],
];

// الجداول المطلوبة
$requiredTables = [
    ['table' => 'vouchers', 'critical' => true],
    ['table' => 'voucher_redemptions', 'critical' => true],
    ['table' => 'voucher_codes', 'critical' => false],
    ['table' => 'broadcast_blacklist', 'critical' => true],
    ['table' => 'blocked_broadcasts', 'critical' => true],
    ['table' => 'system_check_reports', 'critical' => true],
    ['table' => 'broadcast_settings', 'critical' => true],
    ['table' => 'broadcast_errors', 'critical' => false],
    ['table' => 'payment_logs', 'critical' => false],
];

$schemaOk = true;

// فحص الجداول
echo "\n   📋 Required Tables:\n";
foreach ($requiredTables as $req) {
    try {
        $exists = Db::fetchValue(
            "SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = ?",
            [$req['table']]
        );
        
        $passed = ($exists > 0);
        $icon = $passed ? "✅" : ($req['critical'] ? "❌" : "⚠️");
        
        echo "      {$icon} {$req['table']}" . ($passed ? "" : " (MISSING)") . "\n";
        
        $report['details']['schema_checks'][] = [
            'type' => 'table',
            'name' => $req['table'],
            'exists' => $passed,
            'critical' => $req['critical'],
        ];
        
        if (!$passed && $req['critical']) {
            $schemaOk = false;
        }
        
    } catch (Throwable $e) {
        echo "      ❌ {$req['table']}: ERROR\n";
        $schemaOk = false;
    }
}

// فحص الأعمدة
echo "\n   📋 Required Columns:\n";
foreach ($requiredColumns as $req) {
    try {
        $exists = Db::fetchValue(
            "SELECT COUNT(*) FROM information_schema.columns 
             WHERE table_schema = DATABASE() AND table_name = ? AND column_name = ?",
            [$req['table'], $req['column']]
        );
        
        $passed = ($exists > 0);
        $icon = $passed ? "✅" : ($req['critical'] ? "❌" : "⚠️");
        
        echo "      {$icon} {$req['table']}.{$req['column']}" . ($passed ? "" : " (MISSING)") . "\n";
        
        $report['details']['required_columns'][] = [
            'table' => $req['table'],
            'column' => $req['column'],
            'exists' => $passed,
            'critical' => $req['critical'],
        ];
        
        if (!$passed && $req['critical']) {
            $schemaOk = false;
        }
        
    } catch (Throwable $e) {
        echo "      ❌ {$req['table']}.{$req['column']}: ERROR\n";
        $schemaOk = false;
    }
}

$report['summary']['schema'] = $schemaOk ? 'PASS' : 'FAIL';
$textReport[] = ($schemaOk ? "✅" : "❌") . " Schema Check: " . ($schemaOk ? "All required tables/columns exist" : "Missing critical elements");

if (!$schemaOk) {
    $report['details']['recommendations'][] = [
        'check' => 'schema',
        'recommendation' => 'Run migration: mysql -u root -p database < sql/migrations/001_vouchers_and_permissions.sql',
    ];
}

echo "\n";

// ============================================================
// D. فحص توافق الكود
// ============================================================

echo "🔍 D. Code Compatibility Check\n";
echo "─────────────────────────────────────────────────────────────────\n";

$codeIssues = [];
$srcDir = __DIR__ . '/../src';
$libsDir = __DIR__ . '/../libs';

// البحث عن مراجع قديمة محتملة
$searchPatterns = [
    'expires_at' => 'قد يكون هناك خلط بين expires_at و access_expires_at',
    'broadcast_content_type' => 'تم استبداله بـ broadcast_allowed_types',
];

foreach ($searchPatterns as $pattern => $note) {
    // البحث في src/
    $files = glob("{$srcDir}/**/*.php");
    foreach ($files as $file) {
        $content = file_get_contents($file);
        if (preg_match_all("/\b{$pattern}\b/", $content, $matches, PREG_OFFSET_CAPTURE)) {
            foreach ($matches[0] as $match) {
                // حساب رقم السطر
                $lineNum = substr_count(substr($content, 0, $match[1]), "\n") + 1;
                $codeIssues[] = [
                    'file' => str_replace(__DIR__ . '/../', '', $file),
                    'line' => $lineNum,
                    'pattern' => $pattern,
                    'note' => $note,
                ];
            }
        }
    }
}

if (empty($codeIssues)) {
    echo "   ✅ No obvious compatibility issues found\n";
    $report['summary']['code_compatibility'] = 'PASS';
    $textReport[] = "✅ Code Compatibility: No issues found";
} else {
    echo "   ⚠️ Found " . count($codeIssues) . " potential issues:\n";
    foreach (array_slice($codeIssues, 0, 5) as $issue) {
        echo "      • {$issue['file']}:{$issue['line']} - '{$issue['pattern']}'\n";
    }
    $report['summary']['code_compatibility'] = 'WARN';
    $report['details']['code_compatibility'] = $codeIssues;
    $textReport[] = "⚠️ Code Compatibility: " . count($codeIssues) . " potential issues";
}

echo "\n";

// ============================================================
// E. Functional Tests
// ============================================================

echo "🧪 E. Functional Tests\n";
echo "─────────────────────────────────────────────────────────────────\n";

$functionalTestsPassed = 0;
$functionalTestsFailed = 0;

// E1. فحص توليد الأكواد (CodeGenerator)
echo "\n   E1. Code Generation:\n";
try {
    $activationCode = CodeGenerator::generateActivationCode();
    $validFormat = preg_match('/^ACT-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $activationCode);
    
    if ($validFormat) {
        echo "      ✅ Activation code format OK: {$activationCode}\n";
        $functionalTestsPassed++;
        $report['details']['functional_tests'][] = [
            'name' => 'Code Generation - Activation',
            'status' => 'PASS',
            'sample' => $activationCode,
        ];
    } else {
        echo "      ❌ Invalid activation code format: {$activationCode}\n";
        $functionalTestsFailed++;
    }
    
    // كود تحقق
    $verificationCode = CodeGenerator::generateVerificationCode('test_secret');
    $validVerification = (strlen($verificationCode) === 6 && ctype_digit($verificationCode));
    
    if ($validVerification) {
        echo "      ✅ Verification code format OK: {$verificationCode}\n";
        $functionalTestsPassed++;
        $report['details']['functional_tests'][] = [
            'name' => 'Code Generation - Verification',
            'status' => 'PASS',
            'sample' => $verificationCode,
        ];
    } else {
        echo "      ❌ Invalid verification code format: {$verificationCode}\n";
        $functionalTestsFailed++;
    }
    
} catch (Throwable $e) {
    echo "      ❌ Code generation failed: " . $e->getMessage() . "\n";
    $functionalTestsFailed++;
}

// E2. فحص Voucher System
echo "\n   E2. Voucher System:\n";
try {
    // التحقق من وجود class Voucher
    if (class_exists('Voucher')) {
        echo "      ✅ Voucher class loaded\n";
        $functionalTestsPassed++;
        
        // اختبار توليد كود قسيمة
        $voucherCode = Voucher::generateCode();
        $validVoucherFormat = preg_match('/^VCH-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $voucherCode);
        
        if ($validVoucherFormat) {
            echo "      ✅ Voucher code format OK: {$voucherCode}\n";
            $functionalTestsPassed++;
            $report['details']['functional_tests'][] = [
                'name' => 'Voucher Code Generation',
                'status' => 'PASS',
                'sample' => $voucherCode,
            ];
        } else {
            echo "      ❌ Invalid voucher code format: {$voucherCode}\n";
            $functionalTestsFailed++;
        }
    } else {
        echo "      ❌ Voucher class not loaded\n";
        $functionalTestsFailed++;
    }
} catch (Throwable $e) {
    echo "      ❌ Voucher test failed: " . $e->getMessage() . "\n";
    $functionalTestsFailed++;
}

// E3. فحص BroadcastFilter
echo "\n   E3. Broadcast Filter System:\n";
try {
    if (class_exists('BroadcastFilter')) {
        echo "      ✅ BroadcastFilter class loaded\n";
        $functionalTestsPassed++;
        
        // اختبار فحص blacklist
        $cleanCheck = BroadcastFilter::checkBlacklist('Hello world');
        if ($cleanCheck['clean'] === true) {
            echo "      ✅ Blacklist check (clean content) OK\n";
            $functionalTestsPassed++;
        }
        
        // اختبار محتوى محظور (افتراضي)
        $dirtyCheck = BroadcastFilter::checkBlacklist('This contains porn word');
        if ($dirtyCheck['clean'] === false) {
            echo "      ✅ Blacklist check (blocked content) OK\n";
            $functionalTestsPassed++;
        } else {
            echo "      ⚠️ Blacklist might be empty or disabled\n";
            addWarning($report, 'Blacklist', 'Blacklist might be empty - default words not loaded');
        }
        
        $report['details']['functional_tests'][] = [
            'name' => 'BroadcastFilter Blacklist',
            'status' => 'PASS',
            'clean_test' => $cleanCheck['clean'],
            'dirty_test' => !$dirtyCheck['clean'],
        ];
        
    } else {
        echo "      ❌ BroadcastFilter class not loaded\n";
        $functionalTestsFailed++;
    }
} catch (Throwable $e) {
    echo "      ❌ BroadcastFilter test failed: " . $e->getMessage() . "\n";
    $functionalTestsFailed++;
}

// E4. فحص SystemChecker
echo "\n   E4. System Checker:\n";
try {
    if (class_exists('SystemChecker')) {
        echo "      ✅ SystemChecker class loaded\n";
        $functionalTestsPassed++;
        
        // تشغيل فحص سريع
        echo "      ⏳ Running quick system check...\n";
        $checkReport = SystemChecker::runQuick(null, 'verification');
        
        echo "      ✅ System check completed\n";
        echo "         • Total: {$checkReport['stats']['total']}\n";
        echo "         • Passed: {$checkReport['stats']['passed']}\n";
        echo "         • Failed: {$checkReport['stats']['failed']}\n";
        echo "         • Warnings: {$checkReport['stats']['warnings']}\n";
        
        $functionalTestsPassed++;
        $report['details']['functional_tests'][] = [
            'name' => 'SystemChecker Quick Run',
            'status' => $checkReport['stats']['failed'] === 0 ? 'PASS' : 'WARN',
            'stats' => $checkReport['stats'],
        ];
        
    } else {
        echo "      ❌ SystemChecker class not loaded\n";
        $functionalTestsFailed++;
    }
} catch (Throwable $e) {
    echo "      ❌ SystemChecker test failed: " . $e->getMessage() . "\n";
    $functionalTestsFailed++;
}

// E5. فحص Limits System
echo "\n   E5. Limits System:\n";
try {
    if (class_exists('Limits')) {
        $globalLimits = Limits::getGlobalLimits();
        
        if ($globalLimits) {
            echo "      ✅ Limits system working\n";
            echo "         • Daily: " . ($globalLimits['daily'] ?? 'null') . "\n";
            echo "         • Weekly: " . ($globalLimits['weekly'] ?? 'null') . "\n";
            echo "         • Monthly: " . ($globalLimits['monthly'] ?? 'null') . "\n";
            $functionalTestsPassed++;
            $report['details']['functional_tests'][] = [
                'name' => 'Limits System',
                'status' => 'PASS',
                'global_limits' => $globalLimits,
            ];
        } else {
            echo "      ⚠️ No global limits defined\n";
            addWarning($report, 'Limits', 'No global limits defined - using defaults');
        }
    }
} catch (Throwable $e) {
    echo "      ❌ Limits test failed: " . $e->getMessage() . "\n";
    $functionalTestsFailed++;
}

// E6. فحص Broadcast Settings
echo "\n   E6. Broadcast Settings:\n";
try {
    $settings = BroadcastFilter::getAllSettings();
    
    if (!empty($settings)) {
        echo "      ✅ Broadcast settings loaded (" . count($settings) . " settings)\n";
        foreach ($settings as $key => $value) {
            echo "         • {$key}: {$value}\n";
        }
        $functionalTestsPassed++;
        $report['details']['functional_tests'][] = [
            'name' => 'Broadcast Settings',
            'status' => 'PASS',
            'settings' => $settings,
        ];
    } else {
        echo "      ⚠️ No broadcast settings found (will use defaults)\n";
        addWarning($report, 'Broadcast Settings', 'No settings in database - defaults will be used');
    }
} catch (Throwable $e) {
    echo "      ❌ Broadcast settings test failed: " . $e->getMessage() . "\n";
    $functionalTestsFailed++;
}

$totalFunctional = $functionalTestsPassed + $functionalTestsFailed;
$report['summary']['functional_tests'] = $functionalTestsFailed === 0 ? 'PASS' : 
    ($functionalTestsFailed < $totalFunctional / 2 ? 'WARN' : 'FAIL');

$textReport[] = ($functionalTestsFailed === 0 ? "✅" : "⚠️") . 
    " Functional Tests: {$functionalTestsPassed}/{$totalFunctional} passed";

echo "\n";

// ============================================================
// F. فحص ملفات الـ Logs
// ============================================================

echo "📄 F. Log Files Check\n";
echo "─────────────────────────────────────────────────────────────────\n";

$logsDir = __DIR__ . '/../logs';
$logErrors = [];

$logFiles = [
    'public_bot.log',
    'admin_bot.log',
    'error.log',
];

foreach ($logFiles as $logFile) {
    $logPath = "{$logsDir}/{$logFile}";
    if (file_exists($logPath)) {
        // قراءة آخر 100 سطر
        $lines = [];
        $fp = fopen($logPath, 'r');
        if ($fp) {
            $pos = -2;
            $line = '';
            $count = 0;
            
            fseek($fp, 0, SEEK_END);
            $size = ftell($fp);
            
            while ($pos > -$size && $count < 100) {
                fseek($fp, $pos, SEEK_END);
                $char = fgetc($fp);
                if ($char === "\n") {
                    if (!empty($line)) {
                        $lines[] = strrev($line);
                        $count++;
                    }
                    $line = '';
                } else {
                    $line .= $char;
                }
                $pos--;
            }
            fclose($fp);
        }
        
        // البحث عن أخطاء
        foreach ($lines as $line) {
            if (preg_match('/error|exception|fatal/i', $line)) {
                $logErrors[] = [
                    'file' => $logFile,
                    'line' => substr($line, 0, 200),
                ];
            }
        }
        
        echo "   ✅ {$logFile} exists (" . count($lines) . " recent lines read)\n";
    } else {
        echo "   ⚠️ {$logFile} not found\n";
    }
}

if (!empty($logErrors)) {
    echo "   ⚠️ Found " . count($logErrors) . " error entries in logs\n";
    $report['details']['logs_errors'] = array_slice($logErrors, 0, 10);
    $report['summary']['logs'] = 'WARN';
    $textReport[] = "⚠️ Logs: " . count($logErrors) . " errors found";
} else {
    echo "   ✅ No recent errors in logs\n";
    $report['summary']['logs'] = 'PASS';
    $textReport[] = "✅ Logs: Clean";
}

echo "\n";

// ============================================================
// G. توليد التقرير النهائي
// ============================================================

echo "📊 G. Generating Final Report\n";
echo "─────────────────────────────────────────────────────────────────\n";

// حساب الإجمالي
$duration = round((microtime(true) - $startTime) * 1000);
$report['duration_ms'] = $duration;

$overallPass = true;
foreach ($report['summary'] as $check => $status) {
    if ($status === 'FAIL') {
        $overallPass = false;
        break;
    }
}

$report['overall_status'] = $overallPass ? 'PASS' : 'FAIL';

// إضافة الإجمالي للتقرير النصي
$textReport[] = "";
$textReport[] = "═══════════════════════════════════════════════════════════════";
$textReport[] = "                       SUMMARY";
$textReport[] = "═══════════════════════════════════════════════════════════════";
$textReport[] = "";
$textReport[] = "Overall Status: " . ($overallPass ? "✅ PASS" : "❌ FAIL");
$textReport[] = "";
$textReport[] = "Stats:";
$textReport[] = "  • Total Checks: {$report['stats']['total_checks']}";
$textReport[] = "  • Passed: {$report['stats']['passed']}";
$textReport[] = "  • Failed: {$report['stats']['failed']}";
$textReport[] = "  • Warnings: {$report['stats']['warnings']}";
$textReport[] = "  • Duration: {$duration}ms";
$textReport[] = "";

if (!empty($report['details']['recommendations'])) {
    $textReport[] = "Recommendations:";
    foreach ($report['details']['recommendations'] as $rec) {
        $textReport[] = "  • [{$rec['check']}] " . ($rec['recommendation'] ?? $rec['message'] ?? '');
    }
}

$textReport[] = "";
$textReport[] = "═══════════════════════════════════════════════════════════════";

// حفظ الملفات
$jsonPath = "{$reportsDir}/{$timestamp}_verification.json";
$txtPath = "{$reportsDir}/{$timestamp}_verification.txt";

file_put_contents($jsonPath, json_encode($report, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
file_put_contents($txtPath, implode("\n", $textReport));

echo "   ✅ JSON Report: {$jsonPath}\n";
echo "   ✅ Text Report: {$txtPath}\n\n";

// ============================================================
// عرض الملخص النهائي
// ============================================================

echo "═══════════════════════════════════════════════════════════════\n";
echo "                    FINAL SUMMARY\n";
echo "═══════════════════════════════════════════════════════════════\n\n";

echo "Overall Status: " . ($overallPass ? "✅ PASS" : "❌ FAIL") . "\n\n";

echo "Summary by Category:\n";
foreach ($report['summary'] as $check => $status) {
    $icon = match($status) {
        'PASS' => '✅',
        'FAIL' => '❌',
        'WARN' => '⚠️',
        default => '❓',
    };
    echo "   {$icon} {$check}: {$status}\n";
}

echo "\nStats:\n";
echo "   • Total Checks: {$report['stats']['total_checks']}\n";
echo "   • Passed: {$report['stats']['passed']}\n";
echo "   • Failed: {$report['stats']['failed']}\n";
echo "   • Warnings: {$report['stats']['warnings']}\n";
echo "   • Duration: {$duration}ms\n";

if (!empty($report['details']['recommendations'])) {
    echo "\n💡 Recommendations:\n";
    foreach (array_slice($report['details']['recommendations'], 0, 5) as $rec) {
        echo "   • [{$rec['check']}] " . ($rec['recommendation'] ?? $rec['message'] ?? '') . "\n";
    }
}

echo "\n═══════════════════════════════════════════════════════════════\n";

// Exit code based on results
exit($overallPass ? 0 : 1);
