#!/usr/bin/env php
<?php
/**
 * CLI Script لتشغيل فحص النظام
 * 
 * الاستخدام:
 *   php cli/system_check.php [full|quick]
 * 
 * الأمثلة:
 *   php cli/system_check.php full
 *   php cli/system_check.php quick
 */

// التحقق من CLI
if (php_sapi_name() !== 'cli') {
    die("This script can only be run from command line\n");
}

// تعريف الثوابت
define('BOT_STARTED', true);
define('BOT_TYPE', 'cli');

// تحميل الملفات الأساسية
require __DIR__ . '/../libs/bootstrap.php';

// قراءة المعاملات
$type = $argv[1] ?? 'full';

echo "═══════════════════════════════════════════════════════════════\n";
echo "              System Checker - Bot Distribution System\n";
echo "═══════════════════════════════════════════════════════════════\n\n";

echo "🔧 Check Type: {$type}\n";
echo "📅 Started at: " . date('Y-m-d H:i:s') . "\n";
echo "─────────────────────────────────────────────────────────────────\n\n";

try {
    $report = match($type) {
        'quick' => SystemChecker::runQuick(null, 'cli'),
        'custom' => SystemChecker::runCustom(array_slice($argv, 2), null, 'cli'),
        default => SystemChecker::runFull(null, 'cli'),
    };
    
    // عرض النتائج
    echo "📊 RESULTS:\n";
    echo "─────────────────────────────────────────────────────────────────\n\n";
    
    foreach ($report['results'] as $result) {
        $icon = match($result['status']) {
            'pass' => '✅',
            'fail' => '❌',
            'warn' => '⚠️',
            default => '⏭️',
        };
        
        echo "{$icon} [{$result['test']}] {$result['message']}\n";
        
        if (!empty($result['recommendation'])) {
            echo "   💡 Recommendation: {$result['recommendation']}\n";
        }
        echo "\n";
    }
    
    // الملخص
    echo "─────────────────────────────────────────────────────────────────\n";
    echo "📊 SUMMARY:\n";
    echo "─────────────────────────────────────────────────────────────────\n\n";
    
    $stats = $report['stats'];
    echo "   Total Tests:  {$stats['total']}\n";
    echo "   ✅ Passed:    {$stats['passed']}\n";
    echo "   ❌ Failed:    {$stats['failed']}\n";
    echo "   ⚠️ Warnings:  {$stats['warnings']}\n";
    echo "   ⏱️ Duration:  {$report['duration_ms']}ms\n\n";
    
    // التوصيات
    if (!empty($report['recommendations'])) {
        echo "─────────────────────────────────────────────────────────────────\n";
        echo "💡 RECOMMENDATIONS:\n";
        echo "─────────────────────────────────────────────────────────────────\n\n";
        
        foreach ($report['recommendations'] as $rec) {
            echo "   • [{$rec['test']}] {$rec['recommendation']}\n";
        }
        echo "\n";
    }
    
    // ملفات التقرير
    echo "─────────────────────────────────────────────────────────────────\n";
    echo "📁 REPORT FILES:\n";
    echo "─────────────────────────────────────────────────────────────────\n\n";
    
    if (!empty($report['report_file_json'])) {
        echo "   JSON: {$report['report_file_json']}\n";
    }
    if (!empty($report['report_file_txt'])) {
        echo "   TXT:  {$report['report_file_txt']}\n";
    }
    
    echo "\n═══════════════════════════════════════════════════════════════\n";
    
    // Exit code based on results
    exit($stats['failed'] > 0 ? 1 : 0);
    
} catch (Throwable $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
    exit(2);
}
