<?php
/**
 * ============================================================
 * بوت الإدارة - نقطة دخول Webhook
 * ============================================================
 * 
 * هذا الملف يستقبل التحديثات من Telegram ويوجّهها لمعالج الأوامر.
 * 
 * إعداد Webhook:
 *   https://api.telegram.org/bot{TOKEN}/setWebhook?url=https://your-domain.com/admin_bot.php
 * 
 * ملاحظة أمنية:
 *   - فقط ADMIN_IDS المحددون في env.php يمكنهم استخدام هذا البوت
 *   - جميع الرسائل من غير المسؤولين تُرفض
 */

declare(strict_types=1);

// تعريف الثوابت
define('BOT_STARTED', true);
define('BOT_TYPE', 'admin');

// تحميل المكتبات
require __DIR__ . '/libs/bootstrap.php';

// تحميل معالج الأوامر
require __DIR__ . '/src/admin/Commands.php';

// ============================================================
// استقبال ومعالجة التحديث
// ============================================================

try {
    // قراءة التحديث من Telegram
    $input = file_get_contents('php://input');
    $update = json_decode($input, true);
    
    // التحقق من صحة التحديث
    if (!$update || !is_array($update)) {
        Logger::debug('Invalid update received', ['input' => substr($input, 0, 200)]);
        http_response_code(200);
        exit('OK');
    }
    
    // تسجيل التحديث في وضع التصحيح
    Logger::debug('Admin bot update received', ['update_id' => $update['update_id'] ?? 0]);
    
    // استخراج معلومات الرسالة
    $message = $update['message'] ?? $update['edited_message'] ?? null;
    $callbackQuery = $update['callback_query'] ?? null;
    
    // معالجة الرسائل النصية
    if ($message) {
        handleMessage($message);
    }
    // معالجة Callback Query (إذا استُخدمت أزرار في المستقبل)
    elseif ($callbackQuery) {
        handleCallbackQuery($callbackQuery);
    }
    
} catch (Throwable $e) {
    Logger::error('Admin bot error', [
        'message' => $e->getMessage(),
        'file'    => $e->getFile(),
        'line'    => $e->getLine(),
    ]);
}

// دائماً نرجع 200 لتجنب إعادة المحاولة من Telegram
http_response_code(200);
exit('OK');

// ============================================================
// معالجة الرسائل
// ============================================================

/**
 * معالجة رسالة واردة
 * 
 * @param array $message
 */
function handleMessage(array $message): void
{
    $chatId = $message['chat']['id'] ?? 0;
    $fromId = $message['from']['id'] ?? 0;
    $text = trim($message['text'] ?? '');
    $username = $message['from']['username'] ?? null;
    
    // التحقق من أن المرسل أدمن
    if (!isAdmin($fromId)) {
        Telegram::sendMessage($chatId, "⛔ ليس لديك صلاحية استخدام هذا البوت.");
        Logger::event('unauthorized_access', 'user', $fromId, null, null, [
            'username' => $username,
            'text'     => substr($text, 0, 100),
        ]);
        return;
    }
    
    // تجاهل الرسائل الفارغة
    if ($text === '') {
        // قد تكون رسالة وسائط، نتجاهلها حالياً
        // إلا إذا كان هناك حالة بث نشطة
        $state = State::get($fromId);
        if ($state && $state['state'] === State::STATE_BROADCAST_AWAITING_CONTENT) {
            handleBroadcastContent($message, $fromId, $chatId);
        }
        return;
    }
    
    // التحقق من حالات الانتظار
    $state = State::get($fromId);
    if ($state) {
        if (handleStateInput($message, $state, $fromId, $chatId, $text)) {
            return;
        }
    }
    
    // معالجة الأوامر
    if (strpos($text, '/') === 0) {
        handleCommand($text, $fromId, $chatId, $message);
    } else {
        // نص عادي - قد يكون استمراراً لحالة أو خطأ
        Telegram::sendMessage($chatId, "❓ أمر غير معروف.\n\nاستخدم /help لعرض الأوامر المتاحة.");
    }
}

/**
 * معالجة الأوامر
 * 
 * @param string $text النص الكامل
 * @param int $fromId معرف المرسل
 * @param int $chatId معرف المحادثة
 * @param array $message الرسالة الكاملة
 */
function handleCommand(string $text, int $fromId, int $chatId, array $message): void
{
    // تحليل الأمر والمعاملات
    $parts = preg_split('/\s+/', $text, 2);
    $command = strtolower($parts[0]);
    $args = $parts[1] ?? '';
    
    // إزالة اسم البوت من الأمر (مثل /help@bot_username)
    if (strpos($command, '@') !== false) {
        $command = explode('@', $command)[0];
    }
    
    // توجيه الأمر للمعالج المناسب
    AdminCommands::handle($command, $args, $fromId, $chatId, $message);
}

/**
 * معالجة إدخال في حالة انتظار
 * 
 * @param array $message
 * @param array $state
 * @param int $fromId
 * @param int $chatId
 * @param string $text
 * @return bool هل تمت المعالجة؟
 */
function handleStateInput(array $message, array $state, int $fromId, int $chatId, string $text): bool
{
    // التحقق من أمر الإلغاء أولاً
    if (strtolower($text) === '/cancel' || $text === 'إلغاء') {
        State::clear($fromId);
        Telegram::sendMessage($chatId, "❌ تم الإلغاء.");
        return true;
    }
    
    switch ($state['state']) {
        case State::STATE_BROADCAST_AWAITING_CONTENT:
            handleBroadcastContent($message, $fromId, $chatId);
            return true;
            
        case State::STATE_BROADCAST_AWAITING_CONFIRM:
            handleBroadcastConfirm($text, $fromId, $chatId);
            return true;
            
        case State::STATE_AWAITING_REPLY_TEXT:
            handleReplyTextInput($text, $fromId, $chatId, $state);
            return true;
            
        case State::STATE_AWAITING_SUPERDIST_REPLY_TEXT:
            handleSuperDistReplyTextInput($text, $fromId, $chatId, $state);
            return true;
            
        case State::STATE_AWAITING_INPUT:
            return AdminCommands::handlePendingInput($text, $fromId, $chatId, $state);
            
        case 'batch_accounts_awaiting':
            // معالجة بيانات CSV للحسابات
            AdminCommands::handle('/addaccounts_batch', $text, $fromId, $chatId, $message);
            return true;
    }
    
    return false;
}

/**
 * معالجة محتوى البث
 * 
 * @param array $message
 * @param int $fromId
 * @param int $chatId
 */
function handleBroadcastContent(array $message, int $fromId, int $chatId): void
{
    $contentType = 'text';
    $contentText = null;
    $contentFileId = null;
    $caption = null;
    
    // تحديد نوع المحتوى
    if (!empty($message['text'])) {
        $contentType = 'text';
        $contentText = $message['text'];
        
        // فحص طول النص
        if (mb_strlen($contentText) > 4096) {
            Telegram::sendMessage($chatId, "❌ النص طويل جداً!\n\n📝 الحد الأقصى: 4096 حرف\n📏 طول نصك: " . mb_strlen($contentText) . " حرف");
            return;
        }
    } elseif (!empty($message['photo'])) {
        $contentType = 'photo';
        $photos = $message['photo'];
        $contentFileId = end($photos)['file_id']; // أكبر حجم
        $caption = $message['caption'] ?? '';
    } elseif (!empty($message['video'])) {
        $contentType = 'video';
        $contentFileId = $message['video']['file_id'];
        $caption = $message['caption'] ?? '';
    } elseif (!empty($message['document'])) {
        $contentType = 'document';
        $contentFileId = $message['document']['file_id'];
        $caption = $message['caption'] ?? '';
    } elseif (!empty($message['audio'])) {
        $contentType = 'audio';
        $contentFileId = $message['audio']['file_id'];
        $caption = $message['caption'] ?? '';
    } elseif (!empty($message['voice'])) {
        $contentType = 'voice';
        $contentFileId = $message['voice']['file_id'];
    } elseif (!empty($message['sticker'])) {
        $contentType = 'sticker';
        $contentFileId = $message['sticker']['file_id'];
    } elseif (!empty($message['animation'])) {
        $contentType = 'animation';
        $contentFileId = $message['animation']['file_id'];
        $caption = $message['caption'] ?? '';
    } else {
        Telegram::sendMessage($chatId, "❌ نوع المحتوى غير مدعوم.\n\n📝 الأنواع المدعومة: نص، صورة، فيديو، مستند، صوت، ملصق، GIF");
        return;
    }
    
    // فحص طول الوصف (caption)
    if ($caption && mb_strlen($caption) > 1024) {
        Telegram::sendMessage($chatId, "❌ الوصف (caption) طويل جداً!\n\n📝 الحد الأقصى: 1024 حرف\n📏 طول وصفك: " . mb_strlen($caption) . " حرف");
        return;
    }
    
    // حفظ المحتوى والانتقال لحالة التأكيد
    State::setBroadcastContent($fromId, $contentType, $contentText, $contentFileId, $caption);
    
    // جلب عدد المستلمين
    $broadcastData = State::getBroadcastData($fromId);
    $recipientsCount = AdminCommands::getBroadcastRecipientsCount($broadcastData);
    
    $confirmMessage = Replies::get('INFO_BROADCAST_CONFIRM', [
        'content_type'     => $contentType,
        'recipients_count' => $recipientsCount,
    ]);
    
    Telegram::sendMessage($chatId, $confirmMessage);
}

/**
 * معالجة تأكيد البث
 * 
 * @param string $text
 * @param int $fromId
 * @param int $chatId
 */
function handleBroadcastConfirm(string $text, int $fromId, int $chatId): void
{
    $text = mb_strtolower(trim($text));
    
    if ($text === 'تأكيد' || $text === 'confirm' || $text === 'نعم' || $text === 'yes') {
        // بدء البث
        AdminCommands::executeBroadcast($fromId, $chatId);
    } elseif ($text === 'تعديل' || $text === 'edit' || $text === 'modify') {
        // العودة لحالة انتظار المحتوى
        State::changeState($fromId, State::STATE_BROADCAST_AWAITING_CONTENT);
        Telegram::sendMessage($chatId, "📝 أرسل المحتوى الجديد الذي تريد بثّه.\n\nأو أرسل /cancel للإلغاء.");
    } else {
        Telegram::sendMessage($chatId, "⚠️ أرسل:\n• \"تأكيد\" للإرسال\n• \"تعديل\" لتغيير المحتوى\n• /cancel للإلغاء");
    }
}

/**
 * معالجة إدخال نص الرد الافتراضي
 * 
 * @param string $text
 * @param int $fromId
 * @param int $chatId
 * @param array $state
 */
function handleReplyTextInput(string $text, int $fromId, int $chatId, array $state): void
{
    $data = $state['data'] ?? [];
    $key = $data['reply_key'] ?? null;
    
    if (!$key) {
        State::clear($fromId);
        Telegram::sendMessage($chatId, "❌ حدث خطأ. حاول مرة أخرى.");
        return;
    }
    
    Replies::setDefault($key, $text);
    Telegram::sendMessage($chatId, "✅ تم تحديث الرد الافتراضي للمفتاح {$key} بنجاح.");
    
    State::clear($fromId);
}

/**
 * معالجة إدخال نص الرد المخصص للسوبر موزع
 * 
 * @param string $text
 * @param int $fromId
 * @param int $chatId
 * @param array $state
 */
function handleSuperDistReplyTextInput(string $text, int $fromId, int $chatId, array $state): void
{
    $data = $state['data'] ?? [];
    $key = $data['reply_key'] ?? null;
    $superdistId = $data['superdist_id'] ?? null;
    
    if (!$key || !$superdistId) {
        State::clear($fromId);
        Telegram::sendMessage($chatId, "❌ حدث خطأ. حاول مرة أخرى.");
        return;
    }
    
    Replies::setSuperDistReply($superdistId, $key, $text);
    Telegram::sendMessage($chatId, "✅ تم تحديث الرد المخصص للسوبر موزع {$superdistId}.\n\nالمفتاح: {$key}");
    
    State::clear($fromId);
}

/**
 * معالجة Callback Query
 * 
 * @param array $callbackQuery
 */
function handleCallbackQuery(array $callbackQuery): void
{
    $callbackId = $callbackQuery['id'];
    $fromId = $callbackQuery['from']['id'] ?? 0;
    $data = $callbackQuery['data'] ?? '';
    
    // التحقق من الصلاحية
    if (!isAdmin($fromId)) {
        Telegram::answerCallbackQuery($callbackId, "⛔ ليس لديك صلاحية.", true);
        return;
    }
    
    // معالجة البيانات (للاستخدام المستقبلي)
    Telegram::answerCallbackQuery($callbackId);
}
